<?php

namespace App\Http\Controllers;

use App\DataTable\DataTable;
use App\DataTable\DataTableConfiguration;
use App\Exceptions\KitWithMovementsCantBeChangedException;
use App\Exceptions\ProductHasInventoryMovementException;
use App\Helpers;
use App\Http\Controllers\Traits\BulkDeleteProcessor;
use App\Http\Controllers\Traits\BulkOperation;
use App\Http\Controllers\Traits\ImportsData;
use App\Http\Requests\StoreProduct;
use App\Http\Resources\FBAProductInventoryDetailsResource;
use App\Http\Resources\InventoryMovementResource;
use App\Http\Resources\ProductAttributeResource;
use App\Http\Resources\ProductInventoryDetailsResource;
use App\Http\Resources\ProductResource;
use App\Http\Resources\SupplierProductResource;
use App\Models\Attribute;
use App\Models\AttributeGroup;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\ProductAttribute;
use App\Models\ProductInventoryMovement;
use App\Models\ProductPricingTier;
use App\Models\Setting;
use App\Models\SupplierProduct;
use App\Models\Warehouse;
use App\Queries\Product as ProductQuery;
use App\Repositories\SupplierInventoryRepository;
use App\Response;
use App\Validator;
use App\Validators;
use Exception;
use Illuminate\Contracts\Filesystem\FileNotFoundException;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\AnonymousResourceCollection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Throwable;

/**
 * Class ProductController.
 */
class ProductController extends Controller
{
    use BulkDeleteProcessor;
    use bulkOperation;
    use DataTable;
    use ImportsData;

    /**
     * @var string
     */
    protected $model_path = ProductQuery::class;

    /**
     * @var SupplierInventoryRepository
     */
    private $supplierInventory;

    /**
     * ProductController constructor.
     */
    public function __construct(SupplierInventoryRepository $inventory)
    {
        parent::__construct();
        $this->supplierInventory = $inventory;
        //Helpers::saveDoc(new ProductQuery(), "Queries_Product");
        //die();
    }

    public function getQueryBuilder()
    {
        return ProductQuery::query();
    }

    /**
     * Get a product id.
     *
     * @param  int  $id product id
     */
    public function show(int $id): Response
    {
        /**
         * Get product by id with relations Or fail 404 not found.
         */
        $product = ProductQuery::with([])->addRelations(request())->findOrFail($id);

        return $this->response->addData(ProductResource::make($product));
    }

    /**
     * Get a product by SKU.
     */
    public function showBySku(Request $request): Response
    {
        $request->validate(['sku' => 'required']);
        /**
         * Get product by SKU with relations Or fail 404 not found.
         */
        $products = ProductQuery::with([])->addRelations(request())->where('sku', $request->get('sku'))->firstOrFail();

        return $this->response->addData(ProductResource::make($products));
    }

    /**
     * Store new Product.
     *
     *
     * @throws Throwable
     */
    public function store(StoreProduct $request): JsonResponse
    {
        // All inputs as array
        $inputs = $request->all();

        DB::transaction(function () use ($inputs) {
            // New Product and set brand
            $product = new Product($inputs);
            $product->setBrandName($inputs['brand_name'] ?? null, true);

            // Units of measurement
            $defaultWeightUnit = Setting::getValueByKey(Setting::KEY_DEFAULT_WEIGHT_UNIT);
            $defaultDimensionUnit = Setting::getValueByKey(Setting::KEY_DEFAULT_DIMENSION_UNIT);

            $product->dimension_unit = $defaultDimensionUnit;
            $product->case_dimension_unit = $defaultDimensionUnit;
            $product->weight_unit = $defaultWeightUnit;
            $product->case_weight_unit = $defaultWeightUnit;

            $product->save();

            // Bundles
            if ($product->type == 'bundle' && isset($inputs['components'])) {
                $product->setBundleComponents($inputs['components']);
            }

            if (isset($inputs['tags'])) {
                // If the tags input parameter is provided, then set
                // the provided tags (Any tags which are not present in the payload will be removed)
                // If the tags input parameter is not provided, then leave the current tags as they are
                $product->syncTags($inputs['tags']);
            }

            // Pricing
            if (empty($inputs['pricing'])) {
                // We get the default pricing tier
                $defaultPricingTier = ProductPricingTier::with([])->where('is_default', 1)->first();
                $inputs['pricing'] = $defaultPricingTier ? [
                    [
                        'product_pricing_tier_name' => $defaultPricingTier->name,
                        'price' => 0,
                    ],
                ] : [];
            }

            $product->setPricing($inputs['pricing'], false);

            // Source
            $product->setSupplierProducts($inputs['suppliers'] ?? [], false);

            // Taxonomy
            $product->setCategories($inputs['categories'] ?? [], false);
            $product->setAttributeGroups($inputs['attribute_groups'] ?? [], Helpers::OPERATION_SET);
            $product->setProductAttributes($inputs['attributes'] ?? [], false);

            // Images
            $product->addImages($inputs['images'] ?? []);

            // Product Variations
            $product->setVariations($inputs['variations'] ?? []);

            // Load product relations.
            $product->load(DataTableConfiguration::getRequiredRelations(ProductQuery::class));

            $this->response->success(Response::HTTP_CREATED)
                ->setMessage(__('messages.success.create', ['resource' => 'product']))
                ->addData(ProductResource::make($product));
        });

        return $this->response;
    }

    /**
     * Update Product.
     *
     * @param  int  $id product id
     *
     * @throws Throwable
     */
    public function update(StoreProduct $request, int $id): JsonResponse
    {
        // All request inputs as array
        $inputs = $request->all();

        DB::transaction(function () use ($id, $inputs) {
            /**
             * Get Product or fail with 404 status code.
             */
            /** @var Product $product */
            $product = Product::with([])->findOrFail($id);
            $product->setBrandName($inputs['brand_name'] ?? null);
            $product->fill($inputs);

            // If a product has inventory movements, the product type cannot be changed.
            // Throw an exception
            if(
                $product->type !== $product->getOriginal('type') &&
                $product->hasInventoryMovements()
            ) {
                throw new ProductHasInventoryMovementException('Product has inventory movements. Cannot update product type');
            }

            $product->save();

            if (isset($inputs['tags'])) {
                // If the tags input parameter is provided, then set
                // the provided tags (Any tags which are not present in the payload will be removed)
                // If the tags input parameter is not provided, then leave the current tags as they are
                $product->syncTags($inputs['tags']);
            }

            // Pricing
            $product->setPricing($inputs['pricing'] ?? null);

            if ($product->type == 'standard') {
                // Source
                $product->setSupplierProducts($inputs['suppliers'] ?? null);
            } elseif ($product->type == 'bundle' && isset($inputs['components'])) {
                $product->setBundleComponents($inputs['components']);
            } elseif ($product->type == 'kit' && isset($inputs['components'])) {
                try {
                    $product->setKitComponents($inputs['components']);
                } catch (KitWithMovementsCantBeChangedException $e) {
                    return $this->response->error(500)->setMessage($e->getMessage());
                }
            }

            // Set blemished fields
            if ($product->isBlemished() && isset($inputs['blemished'])) {
                $product->updateBlemishedInfo($inputs['blemished']);
            }

            // Taxonomy
            $product->setCategories($inputs['categories'] ?? null);
            // $product->setAttributeGroups( $inputs['attribute_groups'] ?? null, $inputs['attribute_groups_operation'] ?? Helpers::OPERATION_SET );
            $product->setProductAttributes($inputs['attributes'] ?? null);

            // Images
            $product->addImages($inputs['images'] ?? null);

            // Product Variations
            $product->setVariations($inputs['variations'] ?? null, true);
            $product->setSharedChildrenAttributes($inputs['shared_children_attributes'] ?? null);

            // Load product relations.
            $product->load(DataTableConfiguration::getRequiredRelations(ProductQuery::class));

            $this->response->setMessage(__('messages.success.update', ['resource' => 'product', 'id' => $product->sku]))
                ->addData(ProductResource::make($product));
        });

        return $this->response;
    }

    /**
     * Delete Product by ID.
     *
     *
     * @throws Exception
     */
    public function destroy($id): JsonResponse
    {
        /** @var Product $product */
        $product = Product::with([])->findOrFail($id);
        $reasons = $product->delete();

        // check if the product is linked
        if ($reasons and is_array($reasons)) {
            foreach ($reasons as $key => $reason) {
                $this->response->addError($reason, ucfirst(Str::singular($key)).Response::CODE_RESOURCE_LINKED, $key, ['product_id' => $id]);
            }

            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->setMessage(__('messages.product.delete_failed', ['id' => $product->sku]));
        }

        return $this->response->setMessage(__('messages.product.delete_success', ['id' => $product->sku]));
    }

    /**
     * Archive Product by ID.
     */
    public function archive($id): JsonResponse
    {
        $product = Product::with([])->findOrFail($id);

        if ($product->archive()) {
            return $this->response->setMessage(__('messages.success.archive', [
                'resource' => 'product',
                'id' => $product->sku,
            ]));
        }

        return $this->response->addWarning(__('messages.failed.already_archive', [
            'resource' => 'product',
            'id' => $product->sku,
        ]), 'Product'.Response::CODE_ALREADY_ARCHIVED, 'id', ['id' => $id]);
    }

    /**
     * Unarchived product by ID.
     */
    public function unarchived($id): JsonResponse
    {
        $product = Product::with([])->findOrFail($id);

        if ($product->unarchived()) {
            return $this->response->setMessage(__('messages.success.unarchived', [
                'resource' => 'product',
                'id' => $product->sku,
            ]));
        }

        return $this->response->warning()
            ->addWarning(__('messages.failed.unarchived', [
                'resource' => 'product',
                'id' => $product->sku,
            ]), 'Product'.Response::CODE_ALREADY_UNARCHIVED, 'id', ['id' => $id]);
    }

    /**
     * Product Inventory Movements.
     *
     *
     * @return Response|AnonymousResourceCollection
     */
    public function inventoryMovements(Request $request, $productId)
    {
        // only return "table_specifications" if its input is "1"
        if ($request->input('table_specifications') == 1) {
            return $this->response->addData(DataTableConfiguration::getTableSpecifications(ProductInventoryMovement::class));
        }

        // prevent send included and excluded together
        if ($request->has('included') and $request->has('excluded')) {
            return $this->response->error(Response::HTTP_BAD_REQUEST, [__('messages.failed.not_both_include_exclude')])
                ->setMessage(__('messages.failed.not_both_include_exclude'));
        }

        // set default included
        $this->setDefaultIncludedColumns($request, ProductInventoryMovement::class);

        $inventoryMovements = ProductInventoryMovement::with(['warehouse']);

        $inventoryMovements->filter($request);
        $inventoryMovements->addRelations($request);
        $inventoryMovements->addCustomColumns();
        $inventoryMovements->sort($request);

        // to always apply on "and" conjunction
        $inventoryMovements->where('product_id', $productId);

        // paginate with limit per page (default 10)
        $inventoryMovements = DataTableConfiguration::paginate($inventoryMovements);

        return InventoryMovementResource::collectionWithTableSpecifications($inventoryMovements, ProductInventoryMovement::class);
    }

    /**
     * Get Product's Components.
     */
    public function productComponents($productId): AnonymousResourceCollection
    {
        return Product::find($productId)->components;
    }

    /**
     * Get Product's Bundles.
     */
    public function productBundles($productId): AnonymousResourceCollection
    {
        return Product::with(['bundles'])->find($productId)->bundles;
    }

    /**
     * @throws Exception
     */
    public function potentialBundles($productId): Response
    {
        /*
         * for each component, need to get the stock breakdown per warehouse
         * At the same time need to calculate how many bundles the component stock can create (for that warehouse)
         */

        $product = Product::find($productId);
        if ($product === null || $product->type !== 'kit') {
            throw new Exception('Potential bundles can only be calculated for kii products');
        }

        $component_stock = $product->components()->with([
            'productInventory',
        ])->get();

        $potential_bundles = $component_stock->map(function ($component) {
            $component->bundle_data = collect($component->toArray()['product_inventory'])->map(function ($warehouse) use ($component) {
                $statuses = [
                    'inventory_reserved',
                    'inventory_in_transit',
                    'inventory_available',
                ];
                foreach ($statuses as $status) {
                    $data['warehouse_id'] = $warehouse['warehouse_id'];
                    $data[$status] = [
                        'quantity' => $warehouse[$status],
                        'max_bundle_quantity' => intval($warehouse[$status] / $component->pivot->quantity),
                    ];
                }

                return $data;
            });

            return $component;
        });

        return $this->response->addData($potential_bundles);
    }

    /**
     * Get Product's Suppliers.
     */
    public function productSuppliers($productId): AnonymousResourceCollection
    {
        $productSuppliers = SupplierProduct::with([
            'supplier',
            'supplierPricingTiers',
        ])->where('product_id', $productId)->get();

        return SupplierProductResource::collection($productSuppliers)->additional(['status' => __('messages.status_success')]);
    }

    /**
     * Mark Supplier as default supplier of product.
     */
    public function setDefaultSupplier($productId, $supplierId): JsonResponse
    {
        Validator::make([
            'product_id' => $productId,
            'supplier_id' => $supplierId,
        ], [
            'product_id' => 'required|exists:products,id',
            'supplier_id' => 'required|exists:suppliers,id',
        ])->validate();

        SupplierProduct::with([])->where('product_id', $productId)->update(['is_default' => false]);

        SupplierProduct::with([])->updateOrCreate(
            [
                'product_id' => $productId,
                'supplier_id' => $supplierId,
            ],
            ['is_default' => true]
        );

        return $this->response->setMessage(__('messages.product.set_default_supplier', [
            'supplier_id' => $supplierId,
            'product_id' => $productId,
        ]));
    }

    /**
     * Assign attribute groups to the product.
     */
    public function assignAttributeGroups(Request $request, $productId): JsonResponse
    {
        $request->validate([
            'attribute_groups_ids' => 'required|array',
            'attribute_groups_ids.*' => 'required|exists:attribute_groups,id',
        ]);

        $product = Product::with([])->findOrFail($productId);
        $product->attributeGroups()->sync($request->input('attribute_groups_ids', []));

        return $this->response->setMessage(__('messages.attribute_group.assign_success'));
    }

    /**
     * check the possibility of deletion.
     */
    public function isProductDeletable(Request $request): Response
    {
        // validate
        $request->validate([
            'ids' => 'required|array|min:1',
            'ids.*' => 'integer|exists:products,id',
        ]);

        $result = [];
        $ids = array_unique($request->input('ids', []));

        $products = Product::with([])->whereIn('id', $ids)->select('id', 'name', 'sku')->get();
        foreach ($products as $key => $product) {
            $usage = $product->isUsed();

            $result[$key] = $product->only(['id', 'name', 'sku']);
            $result[$key]['deletable'] = $usage === false;
            $result[$key]['reason'] = $usage ?: null;
        }

        return $this->response->addData($result);
    }

    /**
     * Display a list of possible attributes for the specified product.
     */
    public function attributes($productId): Response
    {
        $product = Product::with([
            'productAttributes',
            'categories.attributeGroups.attributes',
            'attributeGroups.attributes',
        ])
            ->findOrFail($productId);

        // attributes from product directly or that have value
        $attributes = collect($product->productAttributes);

        // attributes from categories
        foreach ($product->categories as $category) {
            foreach ($category->attributeGroups as $attributeGroup) {
                $attributes = $attributes->merge($attributeGroup->attributes->map(function ($attribute) use ($attributeGroup) {
                    $attribute->attribute_group = $attributeGroup->only('id', 'name');

                    return $attribute;
                }));
            }
        }

        // attributes from attribute groups
        foreach ($product->attributeGroups as $attributeGroup) {
            $attributes = $attributes->merge($attributeGroup->attributes->map(function ($attribute) use ($attributeGroup) {
                $attribute->attribute_group = $attributeGroup->only('id', 'name');
                //return $attribute;
            }));
        }

        return $this->response->addData(ProductAttributeResource::collection($attributes->unique('id')->keyBy('name')));
    }

    public function attributesGrouped($productId): Response
    {
        $product = Product::with('productAttributes', 'categories.attributeGroups.attributes', 'productAttributes.attributeGroup')
            ->findOrFail($productId);

        $attributes = ['direct' => [], 'groups' => []];

        // attributes from attribute groups
        $this->addGroupsToProductAttributes($product->attributeGroups, $attributes, $product->id);

        // attributes from categories
        foreach ($product->categories as $category) {
            $this->addGroupsToProductAttributes($category->attributeGroups->pluck('attributes')->flatten(), $attributes, $product->id);
        }

        // attributes from product directly or that have value
        $attributes['direct'] = ProductAttributeResource::collection(
            ProductAttribute::with(['attribute'])->where('product_id', $product->id)->whereNull('attribute_group_id')->get()
        );

        return $this->response->addData($attributes);
    }

    private function addGroupsToProductAttributes($attributeGroups, &$attributes, $productId)
    {
        $groupIds = array_unique(array_map(function ($attribute) {
            return $attribute['attribute_group_id'];
        }, $attributeGroups->toArray()));

        foreach ($groupIds as $groupId) {
            $attributeGroup = AttributeGroup::with([])->find(($groupId));
            if (! $attributeGroup) {
                continue;
            }

            $groupAttributes = ProductAttribute::with(['attribute'])
                ->where('attribute_group_id', $groupId)
                ->where('product_id', $productId)
                ->get();

            $attributes['groups'][$attributeGroup->id] = [
                'id' => $attributeGroup->id,
                'name' => $attributeGroup->name,
                'attributes' => ProductAttributeResource::collection($groupAttributes),
            ];
        }

        //        foreach ($attributeGroups as $attributeGroup) {
        //            if (! isset( $attributes['groups'][ $attributeGroup->id ] )) {
        //                $attributes['groups'][ $attributeGroup->id ] = [
        //                'id'         => $attributeGroup->id,
        //                'name'       => $attributeGroup->name,
        //                'attributes' => [],
        //                ];
        //
        //                $groupAttributes = [];
        //
        //                foreach ($attributeGroup->attributes as $attribute) {
        //                    if ($productAttribute = $directProductAttributes->firstWhere( 'id', $attribute->id )) {
        //                        $attribute = $productAttribute;
        //                        $directProductAttributes->forget( $productAttribute->id );
        //                    }
        //
        //                    $groupAttributes[] = ProductAttributeResource::make( $attribute );
        //                }
        //
        //                $attributes['groups'][ $attributeGroup->id ]['attributes'] = $groupAttributes;
        //            }
        //        }
    }

    /**
     * TODO 2022-11-23 Sam susepcts this might not be used
     *                 It was added back during a merge
     *
     * Import products from CSV file.
     *
     *
     * @throws FileNotFoundException
     */
    public function importCSV(Request $request): JsonResponse
    {
        $request->validate([
            'csv_file' => 'required|file',
            'csv_delimiter' => 'nullable',
            'csv_enclosure' => 'nullable',
            'override' => 'nullable|boolean',
        ]);

        /**
         * CSV importer configs with default values.
         */
        $csv_file = $request->file('csv_file');
        $csvDelimiter = stripcslashes($request->input('csv_delimiter', ','));
        $csvEnclosure = stripcslashes($request->input('csv_enclosure', '"'));
        $overrideProducts = boolval($request->input('override', true));

        /**
         * Required columns in csv file.
         */
        $required_columns = [
            'sku',
            'name',
            'barcode',
            'brand',
            'type',
            'weight',
            'weight-unit',
            'length',
            'width',
            'height',
            'dimension-unit',
        ];

        /**
         * Convert csv content to associated array.
         */
        $csvProducts = Helpers::csvToArray($csv_file->get(), $csvDelimiter, $csvEnclosure);

        /**
         * Check if the file has products and required columns exists.
         */
        if (count($csvProducts) && count(array_diff($required_columns, array_keys($csvProducts[0]))) === 0) {
            foreach ($csvProducts as $index => $csvProduct) {
                /**
                 * validate.
                 */
                $validator = Validators::productFromCSV($csvProduct);
                if ($validator->fails()) {
                    foreach ($validator->errors()->toArray() as $key => $errors) {
                        foreach ($errors as $error) {
                            $data = array_merge($error['data'] ?? [], [
                                'index' => $index,
                                'sku' => $csvProduct['sku'] ?? null,
                            ]);
                            $this->response->addWarning($error['message'], $error['code'], "products.$index.$key", $data);
                        }
                    }

                    //Log::error('Product from csv importer ('.($csvProduct['sku'] ?? '').') missing required data: '.$validator->errors()->toJson());
                    continue;
                }
                $product = Product::with([])->firstOrNew(['sku' => $csvProduct['sku']]);
                /**
                 * Check if allow override existed products.
                 */
                if (! $product->exists or ($product->exists and $overrideProducts)) {
                    /**
                     * Fill basic product data
                     * parent_id from parent-sku set NULL if parent sku not found OR empty.
                     * brand_id create new brand if not found.
                     * type, weight_unit and dimension_unit must be valid values.
                     * archived_at if archived true(1) set archived date.
                     */
                    $product->fill($csvProduct);
                    $product->parent_id = empty($csvProduct['parent-sku']) ? null : (Product::with([])->where('sku', $csvProduct['parent-sku'])->first()->id ?? null);
                    $product->weight_unit = $csvProduct['weight-unit'];
                    $defaultDimensionUnit = Setting::getValueByKey(Setting::DEFAULT_DIMENSION_UNIT);
                    $product->dimension_unit = $csvProduct['dimension-unit'] ?? $defaultDimensionUnit;

                    // If a product has inventory movements, the product type cannot be changed.
                    // Throw an exception
                    if(
                        $product->type !== $product->getOriginal('type') &&
                        $product->hasInventoryMovements()
                    ) {
                        throw new ProductHasInventoryMovementException('Product has inventory movements. Cannot update product type');
                    }

                    // set brand and save product
                    $product->setBrandName($csvProduct['brand'] ?? null, true);

                    // archive product
                    if (boolval($csvProduct['archived'] ?? false)) {
                        $product->archive();
                    }

                    // Add description attribute.
                    if (! empty($csvProduct['description'])) {
                        $product->setProductAttributes([
                            [
                                'name' => 'description',
                                'value' => $csvProduct['description'],
                            ],
                        ], false);
                    }

                    // add primary image
                    if (! empty($csvProduct['image-url'])) {
                        $product->addImages([
                            [
                                'url' => $csvProduct['image-url'],
                                'download' => $csvProduct['image-download'] ?? false,
                                'is_primary' => true,
                            ],
                        ]);
                    }

                    // add Tags
                    if (! empty($csvProduct['tags'])) {
                        $product->addTags(explode('|', $csvProduct['tags']), false);
                    }
                }
            }

            return $this->response->setMessage(__('messages.success.import', ['resource' => 'products']));
        }
        if (count($csvProducts)) {
            $diffColumns = array_values(array_diff($required_columns, array_keys($csvProducts[0])));

            return $this->response
                ->error(Response::HTTP_UNPROCESSABLE_ENTITY, array_map(function ($column) {
                    return [
                        $column => [
                            [
                                'message' => "The {$column} column is required",
                                'code' => ucfirst(Str::camel($column)).'IsRequired',
                            ],
                        ],
                    ];
                }, $diffColumns))
                ->setMessage(__('messages.failed.csv_incorrect_format'));
        }

        return $this->response->error(Response::HTTP_UNPROCESSABLE_ENTITY)
            ->addError(__('messages.failed.csv_no_records'), 'CSVFile'.Response::CODE_EMPTY, 'csv_file');
    }

    /**
     * Constants we need it in forms.
     */
    public function constants(): JsonResponse
    {
        return $this->response->addData([
            'product_types' => Product::TYPES,
            'weight_units' => Product::WEIGHT_UNITS,
            'dimension_units' => Product::DIMENSION_UNITS,
            'attribute_types' => Attribute::TYPES,
            'warehouse_types' => Warehouse::TYPES,
            'inventory_movement_type' => InventoryMovement::TYPES,
            'inventory_movement_status' => InventoryMovement::INVENTORY_STATUS,
        ]);
    }

    /**
     * Get product inventory details.
     */
    public function inventory(Request $request, $productId): Response
    {
        $product = Product::with([
            'inWarehousesQuantity', // fetch stock quantity grouped by warehouses
            'inWarehouseTransitQuantity',
            'inWarehousesReservedQuantity',
            'inWarehouseAvailableQuantity',
            'purchaseOrderLinesOpened',
            'availableQuantity',
            'suppliers',
            'suppliers.warehouses',
            'suppliersInventory',
            'productListings',
            'productInventory',
        ])->findOrFail($productId);

        return $this->response->addData(ProductInventoryDetailsResource::make($product));
    }

    /**
     * Get product inventory details.
     */
    public function inventoryFBA($warehouse, $productId): Response
    {
        $warehouse = Warehouse::with(['integrationInstance.salesChannel'])->findOrFail($warehouse);

        $product = Product::with(['productListings' => function ($query) use ($warehouse) {
            $query->where('sales_channel_id', $warehouse->integrationInstance->salesChannel->id);
        }])->findOrFail($productId);

        return $this->response->addData(FBAProductInventoryDetailsResource::make($product));
    }

    public function initialInventory(Request $request, $productId): Response
    {
        // TODO: Implement in future using stock take for product per warehouse
    }

    /**
     * Bulk delete products.
     *
     *
     * @throws Exception
     */
    public function bulkDestroy(Request $request): Response
    {
        $request->validate([
            'filters' => 'required_without:ids',
            'ids' => 'required_without:filters|array|min:1',
        ]);

        if (! $this->isLargeDelete(ProductQuery::class, $request)) {
            return $this->bulkOperation($request, $this->BULK_DELETE);
        }

        // Attempting to deleting a large set of records,
        // we will do it in the background.
        $this->processLargeBulkDelete(ProductQuery::class, $request);

        return $this->response->setMessage('Added to the Queue, it will be process shortly');
    }

    /**
     * Bulk archive products.
     */
    public function bulkArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_ARCHIVE);
    }

    /**
     * @return mixed
     */
    public function bulkUnArchive(Request $request)
    {
        return $this->bulkOperation($request, $this->BULK_UN_ARCHIVE);
    }

    /**
     * {@inheritDoc}
     */
    protected function getModel()
    {
        return ProductQuery::class;
    }

    /**
     * {@inheritDoc}
     */
    protected function getResource()
    {
        return ProductResource::class;
    }

    /**
     * Set default included columns if not set.
     *
     * @param  null  $model
     */
    private function setDefaultIncludedColumns_(Request $request, $model = null)
    {
        // set default model to get its table specifications
        if (is_null($model)) {
            $model = $this->getModel();
        }

        $included = json_decode($request->input('included', '[]'), true);
        $visibleOnly = (bool) $request->input('visible_only', true);

        if (empty($included) && $visibleOnly) {
            $tableColumns = DataTableConfiguration::getTableSpecifications($model)['table_specifications']['columns'];
            $included = collect($tableColumns)->where('default_visible', 1)->pluck('data_name')->all();
            // add default included to request
            $request->request->add(['included' => json_encode($included)]);
        }
    }
}
