<?php

namespace App\Http\Controllers;

use App\DataTable\DataTable;
use App\Http\Controllers\Traits\BulkOperation;
use App\Http\Controllers\Traits\ImportsData;
use App\Http\Requests\CategoryToProductRequest;
use App\Http\Requests\ReassignCategoryToProducts;
use App\Http\Requests\StoreCategory;
use App\Http\Resources\ProductCategoryResource;
use App\Http\Resources\ProductCategoryTreeResource;
use App\Models\ProductCategory;
use App\Models\ProductToCategory;
use App\Response;
use Exception;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\AnonymousResourceCollection;

class ProductCategoryController extends Controller
{
    use BulkOperation, DataTable, ImportsData;

    protected $model_path = ProductCategory::class;

    private $resource = 'category';

    public function indexForTree()
    {
        $productCategories = ProductCategory::buildTree();

        return ProductCategoryTreeResource::collection($productCategories)->additional(['status' => __('messages.status_success')]);
    }

    /**
     * Display a listing of categories based on parent.
     *
     * This function used when the user need to assign and manage categories to products
     */
    public function indexForManage(Request $request): AnonymousResourceCollection
    {
        $parentId = $request->get('parent_id');

        $categories = ProductCategory::with(['childrenWithSub'])
            ->withCount([
                'categoryToProducts',
                'categoryToAttributeGroups',
                'categoryToAttributes',
            ])
            ->where('parent_id', $parentId)
            ->get();

        return ProductCategoryResource::collection($categories)->additional(['status' => __('messages.status_success')]);
    }

    /**
     * Show category.
     */
    public function show($id): Response
    {
        $category = ProductCategory::with('children', 'attributeGroups', 'attributes', 'parent', 'root')->findOrFail($id);

        $subcategoriesIds = $category->getSubcategoriesIds();

        $productsIds = ProductToCategory::with([])
            ->whereIn('category_id', $subcategoriesIds)
            ->limit(10)
            ->pluck('product_id');

        $productsCount = ProductToCategory::with([])
            ->whereIn('category_id', $subcategoriesIds)
            ->count();

        $category->category_to_products_count = $productsCount;
        // products ids
        $category->category_to_products_ids = $productsIds;
        // (children_count - 1) => 1 : is parent category id
        $category->subcategories_count = count($subcategoriesIds) - 1;

        return $this->response->addData(ProductCategoryResource::make($category));
    }

    public function store(StoreCategory $request): JsonResponse
    {
        $category = new ProductCategory($request->all());
        $category->is_leaf = true;
        $category->parent_id = $request->input('parent_id');
        $category->save();

        // store its attribute groups
        $category->attributeGroups()->sync($request->input('attribute_groups', []));
        // store its attributes
        $category->attributes()->sync($request->input('attributes', []));

        return $this->response->success(Response::HTTP_CREATED)
            ->setMessage(__('messages.success.create', ['resource' => $this->resource]))
            ->addData(ProductCategoryResource::make($category));
    }

    public function update(StoreCategory $request, $id): JsonResponse
    {
        $category = ProductCategory::with([])->findOrFail($id);
        $category->fill($request->validated());
        $category->save();

        // sync its attribute groups
        if ($request->has('attribute_groups')) {
            $category->attributeGroups()->sync($request->input('attribute_groups', []));
        }

        // sync its attributes
        if ($request->has('attributes')) {
            $category->attributes()->sync($request->input('attributes', []));
        }

        return $this->response
            ->setMessage(__('messages.success.update', [
                'resource' => $this->resource,
                'id' => $category->name,
            ]))
            ->addData(ProductCategoryResource::make($category));
    }

    /**
     * Remove the specified category.
     *
     *
     * @throws Exception
     */
    public function destroy($id): JsonResponse
    {
        $category = ProductCategory::with([])->findOrFail($id);

        if ($category->delete() === true) {
            return $this->response->setMessage(__('messages.success.delete', [
                'resource' => 'category and its associations',
                'id' => $category->name,
            ]));
        }

        return $this->response->error(Response::HTTP_BAD_REQUEST)
            ->addError(__('messages.category.delete_failed'), 'Category'.Response::CODE_NOT_LEAF, 'id', ['id' => $id]);
    }

    /**
     * Reassigning new category to products.
     */
    public function reassignToProducts(ReassignCategoryToProducts $request): JsonResponse
    {
        $oldCategoryId = $request->input('old_category_id');
        $newCategoryId = $request->input('new_category_id');

        // get "product ids" that assigned to the new category
        $productsInNewCategory = ProductToCategory::with([])
            ->where('category_id', $newCategoryId)
            ->pluck('product_id')
            ->all();

        // reassign products to the new category (is_primary = false) unless products that already assigned to the new category
        $recordCount = ProductToCategory::with([])
            ->where('category_id', $oldCategoryId)
            ->whereNotIn('product_id', $productsInNewCategory)
            ->update([
                'category_id' => $newCategoryId,
                'is_primary' => 0,
            ]);

        // delete product associations with old category that already assigned to the new category
        ProductToCategory::with([])->where('category_id', $oldCategoryId)->whereIn('product_id', $productsInNewCategory)->delete();

        return $this->response->setMessage(__('messages.category.reassign_products'))
            ->addData($recordCount, 'affected_records_count');
    }

    /**
     * Assign and set primary category to a product.
     */
    public function assignToProduct(CategoryToProductRequest $request): JsonResponse
    {
        // set this category as primary category to the product
        $productToCategory = ProductToCategory::with([])->updateOrCreate([
            'category_id' => $request->input('category_id'),
            'product_id' => $request->input('product_id'),
        ], ['is_primary' => $request->input('is_primary', false)]);

        // if is_primary, unset primary category for the product
        if ($productToCategory->is_primary) {
            ProductToCategory::with([])
                ->where('product_id', $productToCategory->product_id)
                ->where('id', '!=', $productToCategory->id)
                ->update(['is_primary' => false]);
        }

        return $this->response->success($productToCategory->wasRecentlyCreated ? Response::HTTP_CREATED : Response::HTTP_OK)
            ->setMessage(__('messages.category.assign_to_product', ['extra' => ($productToCategory->is_primary ? ', and set as primary category' : '')]));
    }

    /**
     * Archive the product category.
     */
    public function archive(int $productCategoryId): JsonResponse
    {
        $productCategory = ProductCategory::with([])->findOrFail($productCategoryId);

        if ($productCategory->archive()) {
            return $this->response
                ->setMessage(__('messages.success.archive', [
                    'resource' => $this->resource,
                    'id' => $productCategory->name,
                ]))
                ->addData(ProductCategoryResource::make($productCategory));
        }

        return $this->response
            ->addWarning(__('messages.failed.already_archive', [
                'resource' => $this->resource,
                'id' => $productCategory->name,
            ]), 'ProductCategory'.Response::CODE_ALREADY_ARCHIVED, 'id', ['id' => $productCategory->id])
            ->addData(ProductCategoryResource::make($productCategory));
    }

    /**
     * Un Archive the product category.
     */
    public function unarchived(int $productCategoryId): JsonResponse
    {
        $productCategory = ProductCategory::with([])->findOrFail($productCategoryId);

        if ($productCategory->unarchived()) {
            return $this->response
                ->setMessage(__('messages.success.unarchived', [
                    'resource' => $this->resource,
                    'id' => $productCategory->name,
                ]))
                ->addData(ProductCategoryResource::make($productCategory));
        }

        return $this->response
            ->addWarning(__('messages.failed.unarchived', [
                'resource' => $this->resource,
                'id' => $productCategory->name,
            ]), 'ProductCategory'.Response::CODE_ALREADY_UNARCHIVED, 'id', ['id' => $productCategory->id])
            ->addData(ProductCategoryResource::make($productCategory));
    }

    /**
     * Bulk delete nominal codes.
     *
     *
     * @throws Exception
     */
    public function bulkDestroy(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_DELETE);
    }

    /**
     * Bulk archive nominal codes.
     */
    public function bulkArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_ARCHIVE);
    }

    /**
     * Bulk unarchived nominal codes.
     *
     *
     * @return mixed
     */
    public function bulkUnArchive(Request $request)
    {
        return $this->bulkOperation($request, $this->BULK_UN_ARCHIVE);
    }

    /**
     * check the possibility of deletion.
     */
    public function isDeletable(Request $request): Response
    {
        // validate
        $request->validate([
            'ids' => 'required|array|min:1',
            'ids.*' => 'integer|exists:product_categories,id',
        ]);

        $ids = array_unique($request->input('ids', []));

        $result = [];
        $productCategories = ProductCategory::with([])->whereIn('id', $ids)->select('id', 'name', 'is_leaf')->get();
        foreach ($productCategories as $key => $productCategory) {
            $result[$key] = $productCategory->only('id', 'name');
            $result[$key]['deletable'] = $productCategory->is_leaf;
            $result[$key]['reason'] = $productCategory->is_leaf ? null : ['productCategory' => __('messages.category.delete_failed')];
        }

        return $this->response->addData($result);
    }

    /**
     * {@inheritDoc}
     */
    protected function getModel()
    {
        return ProductCategory::class;
    }

    /**
     * {@inheritDoc}
     */
    protected function getResource()
    {
        return ProductCategoryResource::class;
    }
}
