<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Traits\BulkOperation;
use App\Http\Requests\StorePaymentType;
use App\Http\Resources\PaymentTypeResource;
use App\Models\PaymentType;
use App\Response;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class PaymentTypeController extends Controller
{
    use BulkOperation;

    protected $model_path = PaymentType::class;

    private $resource = 'payment type';

    public function index()
    {
        // We don't add sales credit payment type
        $types = PaymentType::with([])->get();

        return PaymentTypeResource::collection($types);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StorePaymentType $request): Response
    {
        $paymentType = new PaymentType($request->validated());
        $paymentType->save();

        return $this->response->setMessage(__('messages.success.create', ['resource' => $this->resource]))
            ->addData(PaymentTypeResource::make($paymentType));
    }

    /**
     * Display the specified resource.
     */
    public function show(PaymentType $paymentType): Response
    {
        return $this->response->addData(PaymentTypeResource::make($paymentType));
    }

    public function update(Request $request, $id)
    {
        //TODO: add validation, use FormRequest
        $paymentType = PaymentType::with([])->findOrFail($id);

        $paymentType->fill($request->all())->save();

        return $this->response
            ->setMessage(__('messages.success.update', [
                'resource' => $this->resource,
                'id' => $paymentType->name,
            ]))
            ->addData(PaymentTypeResource::make($paymentType));
    }

    public function updateAccountingMapping(Request $request)
    {
        foreach ($request->get('payment_types') as $paymentTypeIteration) {
            $paymentType = PaymentType::findOrFail($paymentTypeIteration['id']);
            $paymentType->accounting_integration_id = $paymentTypeIteration['accounting_integration_id'];
            $paymentType->accounting_integration_type = $paymentTypeIteration['accounting_integration_type'];
            $paymentType->update();
        }

        return $this->response->success();
    }

    /**
     * Remove the specified resource from storage.
     *
     *
     * @throws Exception
     */
    public function destroy(PaymentType $paymentType): Response
    {
        // Amazon payment type can only be deleted by system.
        if ($paymentType->name === PaymentType::PAYMENT_TYPE_AMAZON) {
            $reasons = [
                'payment_type' => trans_choice('messages.internal_not_deletable', null, [
                    'resource' => 'payment type',
                    'model' => 'payment type('.$paymentType->name.')',
                ]),
            ];
        }
        // Shopify payment type can only be deleted by system
        elseif ($paymentType->name === PaymentType::PAYMENT_TYPE_SHOPIFY) {
            $reasons = [
                'payment_type' => trans_choice('messages.internal_not_deletable', null, [
                    'resource' => 'payment type',
                    'model' => 'payment type('.$paymentType->name.')',
                ]),
            ];
        } else {
            $reasons = $paymentType->delete();
        }

        // check if the payment term is linked
        if ($reasons and is_array($reasons)) {
            foreach ($reasons as $key => $reason) {
                $this->response->addError($reason, ucfirst(Str::singular($key)).Response::CODE_RESOURCE_LINKED, $key, ['payment_type_id' => $paymentType->id]);
            }

            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->setMessage(__('messages.failed.delete', [
                    'resource' => $this->resource,
                    'id' => $paymentType->name,
                ]));
        }

        return $this->response->setMessage(__('messages.success.delete', [
            'resource' => $this->resource,
            'id' => $paymentType->name,
        ]));
    }

    /**
     * check the possibility of deletion.
     */
    public function isDeletable(Request $request): Response
    {
        // validate
        $request->validate([
            'ids' => 'required|array|min:1',
            'ids.*' => 'integer|exists:'.( new $this->model_path )->getTable().',id',
        ]);

        $ids = array_unique($request->input('ids', []));
        $result = [];

        $instances = PaymentType::with([])->withCount('payments')->whereIn('id', $ids)->select('id', 'name')->get();
        foreach ($instances as $key => $instance) {
            $isUsed = $instance->isUsed();

            $result[$key] = $instance->only('id', 'name');
            $result[$key]['deletable'] = ! boolval($isUsed);
            $result[$key]['reason'] = $isUsed ?: null;
        }

        return $this->response->addData($result);
    }

    /**
     * Archive the payment types.
     */
    public function archive(PaymentType $paymentType): Response
    {
        if ($paymentType->archive()) {
            return $this->response->setMessage(__('messages.success.archive', [
                'resource' => $this->resource,
                'id' => $paymentType->name,
            ]))
                ->addData(PaymentTypeResource::make($paymentType));
        }

        return $this->response->warning()
            ->addWarning(__('messages.failed.already_archive', [
                'resource' => $this->resource,
                'id' => $paymentType->name,
            ]), 'PaymentType'.Response::CODE_ALREADY_ARCHIVED, 'id', ['id' => $paymentType->id])
            ->addData(PaymentTypeResource::make($paymentType));
    }

    /**
     * Unarchived the payment types.
     */
    public function unarchived(PaymentType $paymentType): Response
    {
        if ($paymentType->unarchived()) {
            return $this->response
                ->setMessage(__('messages.success.unarchived', [
                    'resource' => $this->resource,
                    'id' => $paymentType->name,
                ]))
                ->addData(PaymentTypeResource::make($paymentType));
        }

        return $this->response->warning()
            ->addWarning(__('messages.failed.unarchived', [
                'resource' => $this->resource,
                'id' => $paymentType->name,
            ]), 'PaymentType'.Response::CODE_ALREADY_UNARCHIVED, 'id', ['id' => $paymentType->id])
            ->addData(PaymentTypeResource::make($paymentType));
    }

    /**
     * bulk archive using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_ARCHIVE);
    }

    /**
     * bulk un archive using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkUnArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_UN_ARCHIVE);
    }

    /**
     * Bulk delete payment types.
     *
     *
     * @throws Exception
     */
    public function bulkDestroy(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_DELETE);
    }

    /**
     * {@inheritDoc}
     */
    protected function getModel()
    {
        return PaymentType::class;
    }

    /**
     * {@inheritDoc}
     */
    protected function getResource()
    {
        return PaymentTypeResource::class;
    }
}
