<?php

namespace App\Http\Controllers;

use App\DataTable\DataTable;
use App\Http\Controllers\Traits\BulkOperation;
use App\Http\Resources\PaymentTermResource;
use App\Models\PaymentTerm;
use App\Response;
use Exception;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class PaymentTermController extends Controller
{
    use BulkOperation, DataTable;

    protected $model_path = PaymentTerm::class;

    private $resource = 'payment term';

    /**
     * Show payment term.
     */
    public function show(PaymentTerm $paymentTerm): JsonResponse
    {
        return $this->response->addData(PaymentTermResource::make($paymentTerm));
    }

    /**
     * Create a new payment term.
     */
    public function store(Request $request): JsonResponse
    {
        $request->validate([
            'name' => 'required|max:255|unique:payment_terms,name',
        ]);

        $paymentTerm = new PaymentTerm();
        $paymentTerm->name = $request->input('name');
        $paymentTerm->save();

        return $this->response->success(Response::HTTP_CREATED)
            ->setMessage(__('messages.success.create', ['resource' => $this->resource]))
            ->addData(PaymentTermResource::make($paymentTerm));
    }

    /**
     * Update a payment term.
     */
    public function update(Request $request, $id): JsonResponse
    {
        $request->validate([
            'name' => 'required|max:255|unique:payment_terms,name,'.$id,
        ]);

        $paymentTerm = PaymentTerm::with([])->findOrFail($id);
        $paymentTerm->name = $request->input('name');
        $paymentTerm->save();

        return $this->response
            ->setMessage(__('messages.success.update', [
                'resource' => $this->resource,
                'id' => $paymentTerm->name,
            ]))
            ->addData(PaymentTermResource::make($paymentTerm));
    }

    /**
     * Archive the nominal code.
     */
    public function archive(PaymentTerm $paymentTerm): JsonResponse
    {
        if ($paymentTerm->archive()) {
            return $this->response->setMessage(__('messages.success.archive', [
                'resource' => $this->resource,
                'id' => $paymentTerm->name,
            ]))
                ->addData(PaymentTermResource::make($paymentTerm));
        }

        return $this->response->warning()
            ->addWarning(__('messages.failed.already_archive', [
                'resource' => $this->resource,
                'id' => $paymentTerm->name,
            ]), 'PaymentTerm'.Response::CODE_ALREADY_ARCHIVED, 'id', ['id' => $paymentTerm->id])
            ->addData(PaymentTermResource::make($paymentTerm));
    }

    /**
     * Unarchived the nominal code.
     */
    public function unarchived(PaymentTerm $paymentTerm): JsonResponse
    {
        if ($paymentTerm->unarchived()) {
            return $this->response
                ->setMessage(__('messages.success.unarchived', [
                    'resource' => $this->resource,
                    'id' => $paymentTerm->name,
                ]))
                ->addData(PaymentTermResource::make($paymentTerm));
        }

        return $this->response->warning()
            ->addWarning(__('messages.failed.unarchived', [
                'resource' => $this->resource,
                'id' => $paymentTerm->name,
            ]), 'PaymentTerm'.Response::CODE_ALREADY_UNARCHIVED, 'id', ['id' => $paymentTerm->id])
            ->addData(PaymentTermResource::make($paymentTerm));
    }

    public function destroy(PaymentTerm $paymentTerm)
    {
        $reasons = $paymentTerm->delete();

        // check if the payment term is linked
        if ($reasons and is_array($reasons)) {
            foreach ($reasons as $key => $reason) {
                $this->response->addError($reason, ucfirst(Str::singular($key)).Response::CODE_RESOURCE_LINKED, $key, ['payment_term_id' => $paymentTerm->id]);
            }

            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->setMessage(__('messages.payment_terms.delete_failed', ['id' => $paymentTerm->name]));
        }

        return $this->response->setMessage(__('messages.payment_terms.delete_success', ['id' => $paymentTerm->name]));
    }

    /**
     * check the possibility of deletion.
     */
    public function isDeletable(Request $request): Response
    {
        // validate
        $request->validate([
            'ids' => 'required|array|min:1',
            'ids.*' => 'integer|exists:'.( new $this->model_path )->getTable().',id',
        ]);

        $ids = array_unique($request->input('ids', []));
        $result = [];

        $instances = PaymentTerm::with([])->whereIn('id', $ids)->select('id', 'name')->get();
        foreach ($instances as $key => $instance) {
            $isUsed = $instance->isUsed();

            $result[$key] = $instance->only('id', 'name');
            $result[$key]['deletable'] = ! boolval($isUsed);
            $result[$key]['reason'] = $isUsed ?: null;
        }

        return $this->response->addData($result);
    }

    /**
     * bulk archive using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_ARCHIVE);
    }

    /**
     * bulk un archive using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkUnArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_UN_ARCHIVE);
    }

    /**
     * Bulk delete payment terms.
     *
     *
     * @throws Exception
     */
    public function bulkDestroy(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_DELETE);
    }

    /**
     * {@inheritDoc}
     */
    protected function getModel()
    {
        return PaymentTerm::class;
    }

    /**
     * {@inheritDoc}
     */
    protected function getResource()
    {
        return PaymentTermResource::class;
    }
}
