<?php

namespace App\Http\Controllers;

use App\DataTable\DataTable;
use App\Http\Controllers\Traits\BulkOperation;
use App\Http\Controllers\Traits\ImportsData;
use App\Http\Requests\StoreNominalCode;
use App\Http\Resources\NominalCodeResource;
use App\Models\NominalCode;
use App\Models\Setting;
use App\Response;
use Exception;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class NominalCodeController extends Controller
{
    use bulkOperation, DataTable, ImportsData;

    protected $model_path = NominalCode::class;

    private $resource = 'nominal code';

    public function show(NominalCode $nominalCode)
    {
        return $this->response->addData(NominalCodeResource::make($nominalCode));
    }

    /**
     * Create a new nominal code.
     */
    public function store(StoreNominalCode $request): JsonResponse
    {
        $nominalCode = new NominalCode($request->all());
        $nominalCode->save();

        return $this->response->success(Response::HTTP_CREATED)
            ->setMessage(__('messages.success.create', ['resource' => $this->resource]))
            ->addData(NominalCodeResource::make($nominalCode));
    }

    /**
     * Update a nominal code.
     */
    public function update(StoreNominalCode $request, $id): JsonResponse
    {
        $nominalCode = NominalCode::with([])->findOrFail($id);
        $nominalCode->fill($request->all());
        $nominalCode->save();

        return $this->response
            ->setMessage(__('messages.success.update', [
                'resource' => $this->resource,
                'id' => $nominalCode->code,
            ]))
            ->addData(NominalCodeResource::make($nominalCode));
    }

    /**
     * Archive the nominal code.
     */
    public function archive(NominalCode $nominalCode): JsonResponse
    {
        if ($nominalCode->archive()) {
            return $this->response
                ->setMessage(__('messages.success.archive', [
                    'resource' => $this->resource,
                    'id' => $nominalCode->code,
                ]))
                ->addData(NominalCodeResource::make($nominalCode));
        }

        return $this->response->warning()
            ->addWarning(__('messages.failed.already_archive', [
                'resource' => $this->resource,
                'id' => $nominalCode->code,
            ]), 'NominalCode'.Response::CODE_ALREADY_ARCHIVED, 'id', ['id' => $nominalCode->id])
            ->addData(NominalCodeResource::make($nominalCode));
    }

    /**
     * Unarchived the nominal code.
     */
    public function unarchived(NominalCode $nominalCode): JsonResponse
    {
        if ($nominalCode->unarchived()) {
            return $this->response
                ->setMessage(__('messages.success.unarchived', [
                    'resource' => $this->resource,
                    'id' => $nominalCode->code,
                ]))
                ->addData(NominalCodeResource::make($nominalCode));
        }

        return $this->response->warning()
            ->addWarning(__('messages.failed.unarchived', [
                'resource' => $this->resource,
                'id' => $nominalCode->code,
            ]), 'NominalCode'.Response::CODE_ALREADY_UNARCHIVED, 'id', ['id' => $nominalCode->id])
            ->addData(NominalCodeResource::make($nominalCode));
    }

    /**
     * Delete nominal code.
     *
     *
     * @throws Exception
     */
    public function destroy(NominalCode $nominalCode): Response
    {
        $reasons = $nominalCode->delete();

        // check if the nominalCode is linked
        if ($reasons and is_array($reasons)) {
            foreach ($reasons as $key => $reason) {
                $this->response->addError($reason, ucfirst(Str::singular($key)).Response::CODE_RESOURCE_LINKED, $key, ['nominal_code_id' => $nominalCode->id]);
            }

            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->setMessage(__('messages.failed.delete', [
                    'resource' => 'nominal code',
                    'id' => $nominalCode->name,
                ]));
        }

        return $this->response->setMessage(__('messages.success.delete', [
            'resource' => 'nominal code',
            'id' => $nominalCode->name,
        ]));
    }

    /**
     * Bulk delete nominal codes.
     *
     *
     * @throws Exception
     */
    public function bulkDestroy(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_DELETE);
    }

    /**
     * Bulk archive nominal codes.
     */
    public function bulkArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_ARCHIVE);
    }

    /**
     * Bulk unarchived nominal codes.
     *
     *
     * @return mixed
     */
    public function bulkUnArchive(Request $request)
    {
        return $this->bulkOperation($request, $this->BULK_UN_ARCHIVE);
    }

    /**
     * check the possibility of deletion.
     */
    public function isDeletable(Request $request): Response
    {
        // validate
        $request->validate([
            'ids' => 'required|array|min:1',
            'ids.*' => 'integer|exists:nominal_codes,id',
        ]);

        $ids = array_unique($request->input('ids', []));

        $result = [];
        $nominalCodes = NominalCode::with([])->whereIn('id', $ids)->select('id', 'name', 'code')->get();
        foreach ($nominalCodes as $key => $nominalCode) {
            $isUsed = $nominalCode->isUsed();

            $result[$key] = $nominalCode->only('id', 'name', 'code');
            $result[$key]['deletable'] = ! boolval($isUsed);
            $result[$key]['reason'] = $isUsed ?: null;
        }

        return $this->response->addData($result);
    }

    /**
     * check the possibility of archiving.
     */
    public function isArchivable(Request $request): Response
    {
        // validate
        $request->validate([
            'ids' => 'required|array|min:1',
            'ids.*' => 'integer|exists:nominal_codes,id',
        ]);

        $ids = array_unique($request->input('ids', []));

        $settings = Setting::with([])->whereIn('key', Setting::KEYS_NC_MAPPING)->get();
        $result = [];
        $nominalCodes = NominalCode::with([])->whereIn('id', $ids)->select('id', 'name', 'code')->get();
        foreach ($nominalCodes as $key => $nominalCode) {
            $archivable = true;
            if ($settings->firstWhere('value', $nominalCode->id) || $settings->where('value', null)->firstWhere('default_value', $nominalCode->id)) {
                $archivable = false;
            }

            $result[$key] = $nominalCode->only('id', 'name', 'code');
            $result[$key]['archivable'] = $archivable;
            $result[$key]['reason'] = $archivable ? null : ['mappingNominalCodes' => __('messages.nominal_code.used_in_mapping', ['id' => $nominalCode->name])];
        }

        return $this->response->addData($result);
    }

    /**
     * {@inheritDoc}
     */
    protected function getModel()
    {
        return NominalCode::class;
    }

    /**
     * {@inheritDoc}
     */
    protected function getResource()
    {
        return NominalCodeResource::class;
    }
}
