<?php

namespace App\Http\Controllers\Magento;

use App\Http\Controllers\Controller;
use App\Http\Resources\Magento\StoreResource;
use App\Jobs\Magento\GetStores;
use App\Models\IntegrationInstance;
use App\Models\Magento\Store as MagentoStore;
use App\Models\StoreMapping;
use App\Response;
use Illuminate\Http\Request;

class StoreController extends Controller
{
    /**
     * Get Magento Stores
     */
    public function getStores(Request $request, IntegrationInstance $integrationInstance): Response
    {
        $stores = MagentoStore::where('integration_instance_id', $integrationInstance->id)->get();

        return $this->response->addData($stores);
    }

    public function getStoreMappings(Request $request, IntegrationInstance $integrationInstance): Response
    {
        $stores = StoreMapping::where('integration_instance_id', $integrationInstance->id)->get();

        return $this->response->addData(StoreResource::collection($stores));
    }

    /**
     * Map stores
     */
    public function mapStores(Request $request, IntegrationInstance $integrationInstance): Response
    {
        $request->validate([
            'mappings' => 'required|array|min:1',
            'mappings.*.sku_store_id' => 'nullable|exists:stores,id',
            'mappings.*.magento_store_id' => 'nullable|exists:magento_stores,id',
        ]);

        foreach ($request->input('mappings') as $mapping) {
            // save mapping
            if ($mapping['sku_store_id']) {
                $paymentMapping = StoreMapping::query()->updateOrCreate(
                    ['integration_instance_id' => $integrationInstance->id,
                        'magento_store_id' => $mapping['magento_store_id'],
                    ],
                    [
                        'integration_instance_id' => $integrationInstance->id,
                        'sku_store_id' => $mapping['sku_store_id'],
                        'magento_store_id' => $mapping['magento_store_id'],
                    ]
                );
            }
        }

        return $this->response->setMessage(__('messages.success.create', ['resource' => 'magento stores']));
    }

    /**
     * Delete the store mapping
     */
    public function destroyPaymentMethodMapping($storeMapping): Response
    {
        $mapping = StoreMapping::query()->findOrFail($storeMapping);
        $mapping->delete();

        return $this->response->setMessage(__('messages.success.delete', ['resource' => 'magento mapping', 'id' => $mapping->id]));
    }

    public function download(Request $request, IntegrationInstance $integrationInstance): Response
    {
        dispatch_sync(new GetStores($integrationInstance));

        return $this->response->setMessage(__('messages.success.download', ['resource' => 'magento stores']));
    }
}
