<?php

namespace App\Http\Controllers;

use App\DataTable\DataTable;
use App\Http\Resources\InventoryMovementResource;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\PurchaseOrder;
use App\Services\InventoryManagement\InventoryManager;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class InventoryManagementController extends Controller
{
    use DataTable;

    /**
     * Get total cost of products in purchase order.
     *
     * $product_total = $quantity * $unit_cost
     * $product_average = $product_total / $total_lines
     * $product_shipping_cost = $purchase_order_shipping * $product_average
     * $product_total_cost = $product_total + $product_shipping_cost
     */
    private function productsTotalCostInPurchaseOrder(PurchaseOrder $purchaseOrder): array
    {
        $products = [];
        $total_lines = 0;
        foreach ($purchaseOrder->purchaseOrderLines as $purchaseOrderLine) {
            $product_total = $purchaseOrderLine->quantity * $purchaseOrderLine->unit_cost;
            $total_lines += $product_total;

            $products[$purchaseOrderLine->product_id] = [
                'purchase_order_line_id' => $purchaseOrderLine->id,
                'total_cost' => $product_total,
            ];
        }

        foreach ($products as $product_id => $product) {
            $product_average = $product['total_cost'] / $total_lines;

            $products[$product_id]['total_cost'] = $product['total_cost'] + ($purchaseOrder->shipping_cost * $product_average);
        }

        return $products;
    }

    /**
     * {@inheritDoc}
     */
    protected function getModel()
    {
        return InventoryMovement::class;
    }

    /**
     * {@inheritDoc}
     */
    protected function getResource()
    {
        return InventoryMovementResource::class;
    }

    public function tallies(Request $request): JsonResponse
    {
        $warehouse_id = $request->get('warehouse_id');
        $inventoryStatus = $request->get('inventory_status');
        $product_id = $request->get('product_id');

        $inventoryManager = new InventoryManager($warehouse_id, Product::findOrFail($product_id));
        $tallies = $inventoryManager->getTallies($inventoryStatus);

        $responseData = [
            'data' => $tallies,
        ];

        return new JsonResponse($responseData);
    }
}
