<?php

namespace App\Http\Controllers;

use App\Http\Requests\ForecastPurchaseOrdersRequest;
use App\Http\Resources\InventoryForecastResource;
use App\Http\Resources\PurchaseOrderResource;
use App\Repositories\InventoryForecastRepository;
use App\Response;
use App\Services\InventoryForecasting\ForecastManager;
use App\Services\InventoryForecasting\PurchaseOrderBuilder;
use Illuminate\Http\Resources\Json\AnonymousResourceCollection;
use Throwable;

/**
 * Class InventoryForecastController.
 */
class InventoryForecastController extends Controller
{

    /**
     * InventoryForecastController constructor.
     */
    public function __construct(
        private readonly InventoryForecastRepository $cache,
        private readonly ForecastManager             $manager,
        private readonly PurchaseOrderBuilder    $builder,
    )
    {
        parent::__construct();
    }

    /**
     * @return AnonymousResourceCollection
     */
    public function index(): AnonymousResourceCollection
    {
        // Get the inventory projection for today
        return InventoryForecastResource::collection(
            $this->cache->getForecastsForToday()
        );
    }

    public function status(): Response
    {
        $tracking = $this->cache->getTracking();
        return $this->response->setData($tracking);
    }

    public function getTotalCost(): array
    {
        return $this->cache->getTotalCost();
    }

    /**
     * @return Response
     */
    public function refresh(): Response
    {
        try{
            $this->manager->forecast();
            return $this->response->setMessage(
                'Forecasting will be refreshed shortly.'
            );
        }catch (Throwable){
            return $this->response->error()
                ->addError(__('messages.failed.refresh_forecasting'), 'ForecastingInventoryFailed', 'forecasting_refresh_failed');
        }
    }

    public function createPurchaseOrders(ForecastPurchaseOrdersRequest $request): AnonymousResourceCollection|Response
    {
        try{
            return PurchaseOrderResource::collection(
                $this->builder->create($request->validated()['purchase_orders'])
            );
        }catch (Throwable){
            return $this->response->error()
                ->addError(__('messages.failed.create_forecasting_purchase_orders'),
                    'ForecastingPurchaseOrdersFailed',
                    'purchase_orders_creation_failed'
                );
        }

    }
}
