<?php

namespace App\Http\Controllers;

use App\DataTable\DataTable;
use App\DataTable\DataTableConfiguration;
use App\Exceptions\InsufficientStockException;
use App\Http\Controllers\Traits\BulkOperation;
use App\Http\Requests\StoreInventoryAssembly;
use App\Http\Resources\InventoryAssemblyResource;
use App\Models\InventoryAssembly;
use App\Models\Product;
use App\Response;
use App\Services\InventoryAssembly\InventoryAssemblyManager;
use Carbon\Carbon;
use Exception;
use Illuminate\Http\Request;
use Throwable;

class InventoryAssemblyController extends Controller
{
    use BulkOperation, DataTable;

    protected $model_path = InventoryAssembly::class;

    protected $resource = 'inventory assembly';

    protected function getModel(): string
    {
        return InventoryAssembly::class;
    }

    protected function getResource(): string
    {
        return InventoryAssemblyResource::class;
    }

    public function show(InventoryAssembly $assembly): Response
    {
        $assembly->load(DataTableConfiguration::getRequiredRelations(InventoryAssembly::class));

        return $this->response->addData(InventoryAssemblyResource::make($assembly));
    }

    /**
     * @throws Exception
     */
    public function destroy(InventoryAssembly $assembly): Response
    {
        // Delete the inventory assembly
        try {
            $assembly->delete();
        } catch (InsufficientStockException $e) {
            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->addError(__('messages.warehouse.transfer_fifo_used', [
                    'extra' => '',
                    'sku' => $e->productSku,
                    'product_id' => $e->productId,
                ]), 'Product'.Response::CODE_UNACCEPTABLE, 'product', ['product_id' => $e->productId]);
        }

        /*
         * Deletion is currently leaving unused FIFO layers hanging
         */

        return $this->response->setMessage(__('messages.success.delete', [
            'resource' => $this->resource,
            'id' => $assembly->id,
        ]));
    }

    /**
     * check the possibility of deletion.
     */
    public function isDeletable(Request $request): Response
    {
        // validate
        $request->validate([
            'ids' => 'required|array|min:1',
            'ids.*' => 'integer|exists:inventory_assemblies,id',
        ]);

        $ids = array_unique($request->input('ids', []));

        // All inventory assemblies are deletable,
        // only reverses inventory movements.

        $result = [];
        $assemblies = InventoryAssembly::with([])->whereIn('id', $ids)->select('id')->get();
        foreach ($assemblies as $key => $assembly) {
            $result[$key] = $assembly->only('id');
            $result[$key]['deletable'] = true;
            $result[$key]['reason'] = null;
        }

        return $this->response->addData($result);
    }

    /**
     * Assemble Inventory.
     *
     *
     * @throws Throwable
     */
    public function store(StoreInventoryAssembly $request): Response
    {
        try {
            /** @var Product $product */
            $product = Product::with([])->findOrFail($request->input('product_id'));
            $inventoryAssemblyManager = (new InventoryAssemblyManager($product, $product->components));

            $assemble = $request->action == InventoryAssembly::INVENTORY_ASSEMBLY_ACTION_ASSEMBLE;
            if ($assemble) {
                $assembly = $inventoryAssemblyManager->assemble(quantity: $request->quantity, assemblyDate: Carbon::parse($request->action_date), product: null, warehouse_id: $request->warehouse_id);
            } else {
                $assembly = $inventoryAssemblyManager->disassemble(quantity: $request->quantity, assemblyDate: Carbon::parse($request->action_date), product: null, warehouse_id: $request->warehouse_id);
            }
        } catch (Exception $e) {
            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->addError($e->getMessage(), 'InventoryAssemblyCreateFailed', 'id');
        }

        return $this->response
            ->setMessage(__('messages.inventory.assembly_created'))
            ->addData(InventoryAssemblyResource::make($assembly));
    }

    public function archive(InventoryAssembly $assembly): Response
    {
        if ($assembly->archive()) {
            return $this->response
                ->setMessage(__('messages.success.archive', [
                    'resource' => $this->resource,
                    'id' => $assembly->id,
                ]))
                ->addData(InventoryAssemblyResource::make($assembly));
        }

        return $this->response->warning()
            ->addWarning(__('messages.failed.already_archive', [
                'resource' => $this->resource,
                'id' => $assembly->id,
            ]), 'InventoryAssembly'.Response::CODE_ALREADY_ARCHIVED, 'id', ['id' => $assembly->id])
            ->addData(InventoryAssemblyResource::make($assembly));
    }

    public function unarchived(InventoryAssembly $assembly): Response
    {
        if ($assembly->unarchived()) {
            return $this->response
                ->setMessage(__('messages.success.unarchived', [
                    'resource' => $this->resource,
                    'id' => $assembly->id,
                ]))
                ->addData(InventoryAssemblyResource::make($assembly));
        }

        return $this->response
            ->addWarning(__('messages.failed.unarchived', [
                'resource' => $this->resource,
                'id' => $assembly->id,
            ]), 'InventoryAssembly'.Response::CODE_ALREADY_UNARCHIVED, 'id', ['id' => $assembly->id])
            ->addData(InventoryAssemblyResource::make($assembly));
    }

    /**
     * Bulk delete assemblies.
     *
     *
     * @throws Exception
     */
    public function bulkDestroy(Request $request): Response
    {
        try {
            return $this->bulkOperation($request, $this->BULK_DELETE);
        } catch (InsufficientStockException $e) {
            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->addError(__('messages.warehouse.transfer_fifo_used', [
                    'extra' => '',
                    'sku' => $e->productSku,
                    'product_id' => $e->productId,
                ]), 'Product'.Response::CODE_UNACCEPTABLE, 'product', ['product_id' => $e->productId]);
        }
    }

    /**
     * bulk archive using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_ARCHIVE);
    }

    /**
     * bulk un archive using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkUnArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_UN_ARCHIVE);
    }
}
