<?php

namespace App\Http\Controllers;

use App\DataTable\DataTableConfiguration;
use App\DataTable\DataTableResource;
use App\Exporters\ListingExporter;
use App\Http\Requests\DownloadListingsRequest;
use App\Http\Requests\StoreIntegrationInstance;
use App\Http\Resources\IntegrationInstanceResource;
use App\Http\Resources\IntegrationResource;
use App\Http\Resources\PreviewSyncInventoryResource;
use App\Http\Resources\SalesOrderResource;
use App\Integrations\Listing;
use App\Integrations\Listings\ProductMapper;
use App\Jobs\BulkCreateSKUOrdersFromSalesChannelOrders;
use App\Jobs\CacheProductListingPriceJob;
use App\Jobs\GenerateCacheProductListingQuantityJob;
use App\Jobs\DataTableExportJob;
use App\Jobs\DeleteIntegrationInstanceJob;
use App\Jobs\Magento\GetCustomerGroupsJob;
use App\Jobs\Magento\GetProductAttributesJob;
use App\Jobs\Magento\GetProducts as GetMagentoProducts;
use App\Jobs\ShipStation\GetShippingServices;
use App\Jobs\ShipStation\SeedShippingServiceMapping;
use App\Jobs\Shopify\GenerateCreateSalesOrderFromShopifyOrderJobJobs;
use App\Jobs\Shopify\ShopifyGetLocationsJob;
use App\Jobs\Shopify\ShopifyGetOrdersJob;
use App\Jobs\Shopify\ShopifyGetProducts as GetShopifyProducts;
use App\Models\DataImportMapping;
use App\Repositories\IntegrationInstanceRepository;
use App\Services\ShippingProvider\ShipStationManager;
use App\Models\Integration;
use App\Models\IntegrationInstance;
use App\Models\IntegrationShippingMethod;
use App\Models\ProductListing;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\Shopify\ShopifyOrder;
use App\Models\Warehouse;
use App\Response;
use App\Services\Accounting\AccountingTransactionManager;
use Carbon\Carbon;
use Facades\App\Services\Shopify\Orders\Actions\ShopifyDownloadOrder;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\AnonymousResourceCollection;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Laravel\Telescope\Telescope;
use Modules\ShipMyOrders\Entities\ShipMyOrdersIntegrationInstance;
use Modules\ShipMyOrders\Jobs\ShipMyOrdersInitializeWarehouseJob;
use Symfony\Component\HttpFoundation\BinaryFileResponse;

class IntegrationInstanceController extends Controller
{
    /**
     * View integrations and its instances.
     */
    public function integrations($all = false): Response
    {
        $query = Integration::with(['integrationInstances', 'integrationInstances.salesChannel'])->filter();
        $integrations = $all ? $query->get() : $query->where('name', '!=', Integration::NAME_SKU_IO)->get();

        return $this->response->addData(IntegrationResource::collection($integrations))->addAdditional('automated_warehouses', Warehouse::getAutomatedWarehouses());
    }

    /**
     * View an integration instance.
     */
    public function show(IntegrationInstance $integrationInstance): Response
    {
        $integrationInstance->load('integration');

        return $this->response->addData(IntegrationInstanceResource::make($integrationInstance))->addAdditional('automated_warehouses', Warehouse::getAutomatedWarehouses());
    }

    /**
     * Create a new integration instance.
     */
    public function store(StoreIntegrationInstance $request): Response
    {
        $inputs = $request->validated();

        return DB::transaction(function () use ($inputs) {
            if (empty($inputs['name'])) {
                $inputs['name'] = $inputs['integration_settings']['settings']['connectionName'];
            }

            //Tax settings for sales channel
            //      if (Integration::findOrFail($inputs['integration_id'])->integration_type === Integration::TYPE_SALES_CHANNEL
            //      && is_null(@$inputs['integration_settings']['settings']['isTaxInclude'])) {
            //        $inputs['integration_settings']['settings']['isTaxIncluded'] = true;
            //      }

            switch ($inputs['name']) {
                case Integration::NAME_SHIPMYORDERS:
                    $integrationInstance = new ShipMyOrdersIntegrationInstance($inputs);
                    break;
                default:
                    $integrationInstance = new IntegrationInstance($inputs);
                    break;
            }

            $integration_settings = $inputs['integration_settings'];

            /** @var Integration $integration */
            $integration = Integration::query()->findOrFail($inputs['integration_id']);
            if ($integration->isSalesChannel()) {
                $integration_settings['emailCustomers'] = false;
            }

            $startOfMonth = Carbon::now()->startOfMonth()->toDateTimeString();
            $integration_settings['orders']['download']['open_start_date'] = $startOfMonth;
            $integration_settings['orders']['download']['closed_start_date'] = $startOfMonth;
            $integrationInstance->integration_settings = $integration_settings;
            $integrationInstance->save();
            $integrationInstance->setOrderStartDate();

            if ($integrationInstance->integration->integration_type == Integration::TYPE_SALES_CHANNEL
                && ! $integrationInstance->isMagento()
            ) {
                $storeId = $inputs['integration_settings']['settings']['store']['id'];
                $integrationInstance->salesChannel()->create(['store_id' => $storeId]);
            }

            if ($integrationInstance->integration->name == Integration::NAME_SHIPSTATION) {
                dispatch_sync(new GetShippingServices($integrationInstance));
                dispatch_sync(( new SeedShippingServiceMapping($integrationInstance) )->onConnection('sync'));

                $shipStationManager = new ShipStationManager();
                $shipStationManager->createWebhooks($integrationInstance);
            }

            if ($integrationInstance->isShopify()) {
                try {
                    // Create initial listing -> product mapping
                    $dataImportMapping = new DataImportMapping();

                    $dataImportMapping->integration_instance_id = $integrationInstance->id;
                    $dataImportMapping->model = 'listings';
                    $dataImportMapping->mapping = json_decode('[
                              {
                                "listing_field": "sku",
                                "sku_field": "sku",
                                "parsers": []
                              },
                              {
                                  "listing_field": "barcode",
                                "sku_field": "barcode",
                                "parsers": []
                              },
                              {
                                  "listing_field": "variant_title",
                                "sku_field": "name",
                                "parsers": []
                              },
                              {
                                  "listing_field": "price",
                                "sku_field": "price.Retail.value",
                                "parsers": []
                              },
                              {
                                  "listing_field": "unit_cost",
                                "sku_field": "unit_cost",
                                "parsers": []
                              },
                              {
                                  "listing_field": "vendor",
                                "sku_field": "default_supplier",
                                "parsers": []
                              },
                              {
                                  "listing_field": "weight",
                                "sku_field": "weight",
                                "parsers": []
                              },
                              {
                                  "listing_field": "weight_unit",
                                "sku_field": "weight_unit",
                                "parsers": []
                              },
                              {
                                  "listing_field": "tags",
                                "sku_field": "tags",
                                "parsers": []
                              },
                              {
                                  "listing_field": "image1",
                                "sku_field": "image",
                                "parsers": []
                              },
                              {
                                  "listing_field": "image2",
                                "sku_field": "other_images",
                                "parsers": []
                              },
                              {
                                  "listing_field": "image3",
                                "sku_field": "other_images",
                                "parsers": []
                              },
                              {
                                  "listing_field": "image4",
                                "sku_field": "other_images",
                                "parsers": []
                              },
                              {
                                  "listing_field": "image5",
                                "sku_field": "other_images",
                                "parsers": []
                              },
                              {
                                  "listing_field": "image6",
                                "sku_field": "other_images",
                                "parsers": []
                              },
                              {
                                  "listing_field": "image7",
                                "sku_field": "other_images",
                                "parsers": []
                              },
                              {
                                  "listing_field": "image8",
                                "sku_field": "other_images",
                                "parsers": []
                              },
                              {
                                  "listing_field": "image9",
                                "sku_field": "other_images",
                                "parsers": []
                              },
                              {
                                  "listing_field": "image10",
                                "sku_field": "other_images",
                                "parsers": []
                              },
                              {
                                  "listing_field": "image11",
                                "sku_field": "other_images",
                                "parsers": []
                              },
                              {
                                  "listing_field": "image12",
                                "sku_field": "other_images",
                                "parsers": []
                              },
                              {
                                  "listing_field": "image13",
                                "sku_field": "other_images",
                                "parsers": []
                              }
                        ]', true);

                    $dataImportMapping->save();

                    // download shopify locations
                    dispatch_sync((new ShopifyGetLocationsJob($integrationInstance))->onConnection('sync'));
                    if (config('app.env') === 'production') {
                        // create inventory_items/delete webhook
                        /**
                         * Auto webhook setup when creating integration instance is
                         * only supported on production.
                         * Kalvin
                         */
                        Artisan::call('sku:shopify:create-webhooks', ['integrationInstance' => $integrationInstance->id]);
                    }
                } catch (\Throwable $exception) {
                    DB::rollBack();

                    return $this->response->error(Response::HTTP_BAD_REQUEST)->addError("Shopify Failed Message: {$exception->getMessage()}", Response::CODE_INTEGRATION_API_ERROR, 'connection_settings');
                }
            }

            if ($integrationInstance->isMagento()) {
                dispatch_sync(new GetCustomerGroupsJob($integrationInstance));
                dispatch_sync(new GetProductAttributesJob($integrationInstance));
            }

            if (isset($inputs['integration_settings']['pricing']) && ! empty($masterPrice = $inputs['integration_settings']['pricing']['masterOfPrice']['name'])) {
                ProductListing::with([])->where('sales_channel_id', $integrationInstance->salesChannel->id)->update(['master_of_price' => $masterPrice]);
            }

            if ($integrationInstance->integration->name == Integration::NAME_SHIPMYORDERS) {
                /*
                 * This job will create a Warehouse for ShipMyOrders, but must queued
                 * after the response is sent to the client, as this operation is wrapped
                 * in a transaction. If the job gets picked up before the transaction is
                 * complete, the integration instance may not yet exist in the database.
                 * We could work around this if we didn't create it as a job, but this
                 * seems to already be the standard way of doing things.
                 *
                 * Technically the job could fail the lookup, but it would at least show up
                 * as a failed job and not cause an unhelpful error for the client.
                 */
                ShipMyOrdersInitializeWarehouseJob::dispatchAfterResponse($integrationInstance);
            }

            // For veracore, we covert a linked direct warehouse to a 3pl warehouse
            if($integrationInstance->isVeracore()) {
                /** @var Warehouse $linkedWarehouse */
                $linkedWarehouse = Warehouse::query()->findOrFail($integrationInstance->integration_settings['linked_warehouse_id']);
                if($linkedWarehouse->type === Warehouse::TYPE_DIRECT){
                    $linkedWarehouse->type = Warehouse::TYPE_3PL;
                }
                $linkedWarehouse->order_fulfillment = SalesOrderFulfillment::TYPE_VERACORE;
                $linkedWarehouse->save();
            }

            return $this->response->addData(IntegrationInstanceResource::make($integrationInstance))->addAdditional('automated_warehouses', Warehouse::getAutomatedWarehouses(true));
        });
    }

    /**
     * Update an integration instance.
     * @throws \Throwable
     */
    public function update(StoreIntegrationInstance $request, IntegrationInstance $integrationInstance): Response
    {
        $inputs = $request->validated();

        if (empty($inputs['name']) && ! empty($inputs['integration_settings']['settings']['connectionName'])) {
            $inputs['name'] = $inputs['integration_settings']['settings']['connectionName'];
        }

        // check pricing/inventory changed
        $cachePricing = false;
        $cacheQty = false;
        $iSettings = $integrationInstance->integration_settings;

        if (empty($iSettings['pricing']) ||
         (isset($iSettings['pricing'], $inputs['integration_settings']['pricing']) &&
           json_encode($iSettings['pricing']) != json_encode($inputs['integration_settings']['pricing']))) {
            $cachePricing = isset($inputs['integration_settings']['pricing']['masterOfPrice']['name']);
        }

        if ($integrationInstance->isSalesChannel()) {

            $inventorySettingsName = 'inventory';

            // Recalculate cache for listing quantity if the inventory settings have changed
            $cacheQty = isset($inputs['integration_settings']) && (array_map('json_encode', @$iSettings[$inventorySettingsName] ?? []) !== array_map('json_encode', $inputs['integration_settings'][$inventorySettingsName]));
        }

        if ($integrationInstance->integration->name == Integration::NAME_SHIPMYORDERS) {
            $warehouses = Warehouse::where('integration_instance_id', $integrationInstance->id)->get()
                ->pluck('id')->toArray();

            if ($inputs['integration_settings']['settings']['automateFulfillment']) {
                $inputs['integration_settings']['settings']['fulfillment']['automatedWarehousesIds'] = $warehouses;
            }
        }

        DB::transaction(function () use ($integrationInstance, $inputs) {
            $integrationInstance->update($inputs);
            app(AccountingTransactionManager::class)->syncAccountingTransactionsSalesOrderIsSyncEnabledStatus($integrationInstance);
        });

        $integrationInstance->setOrderStartDate();

        if ($integrationInstance->integration->integration_type == Integration::TYPE_SALES_CHANNEL) {
            if (isset($inputs['integration_settings']) && isset($inputs['integration_settings']['settings']['store']['id'])) {
                $integrationInstance->salesChannel->update(['store_id' => $inputs['integration_settings']['settings']['store']['id']]);
            }
        }

        // cache price and quantity
        if ($integrationInstance->salesChannel) {
            if ($cacheQty) {
                GenerateCacheProductListingQuantityJob::dispatchSync($integrationInstance);
            }

            if ($cachePricing) {
                CacheProductListingPriceJob::dispatchSync($integrationInstance);
            }
        }

        return $this->response->addData(IntegrationInstanceResource::make($integrationInstance))->addAdditional('automated_warehouses', Warehouse::getAutomatedWarehouses(true));
    }

    public function downloadListings(IntegrationInstance $integrationInstance)
    {
        if ($integrationInstance->isShopify()) {
            dispatch(new GetShopifyProducts($integrationInstance, [], false))->onQueue('sales-channels');

            return $this->response->setMessage('Shopify Listings download added to the Queue, it will be processed shortly.');
        } elseif ($integrationInstance->isMagento()) {
            dispatch(new GetMagentoProducts($integrationInstance))->onQueue('sales-channels');

            return $this->response->setMessage('Magento Listings download added to the Queue, it will be processed shortly.');
        } else {
            return $this->response->setMessage('This integration does not support downloading listings.');
        }
    }

    /**
     * Delete an integration instance.
     */
    public function destroy(IntegrationInstance $integrationInstance): Response
    {
        DeleteIntegrationInstanceJob::dispatch($integrationInstance)->onQueue('sales-channels');

        return $this->response->setMessage(__('messages.success.queued'));
    }

    /**
     * Lookup products in sales channel.
     *
     *
     * @return Response|AnonymousResourceCollection
     */
    public function salesChannelProducts(Request $request, IntegrationInstance $integrationInstance)
    {
        /** @var Model $modelPath */
        $modelPath = $integrationInstance->integration->getProductsModelPath();

        $modelBuilder = $modelPath::with(['productListing.product']);

        // only return "table_specifications" if its input is "1"
        if ($request->input('table_specifications') == 1) {
            return $this->response->addData(DataTableConfiguration::getTableSpecifications($modelBuilder->getModel()->dataTableKey));
        }

        $filters = json_decode($request->get('filters'), true);
        if(!empty($filters) && !empty($filters['filterSet'])){
            // Handle product listing separately.
            $filterSet = $filters['filterSet'];
            $productListingFilter = collect($filterSet)
                ->filter(fn($filter) => str_starts_with($filter['column'], 'product'));

            if($productListingFilter->isNotEmpty() && method_exists($modelPath, 'scopeJoinProductListings')){
                $modelBuilder->joinProductListings();
            }
        }


        $modelBuilder->filter($request);
        $modelBuilder->sort($request);

        // to force apply on "and" conjunction
        $modelBuilder->where('integration_instance_id', $integrationInstance->id);
        $modelBuilder->addSelect(
            (new $modelPath)->getTable().'.*'
        );

        // paginate with limit per page (default 10)
        $products = DataTableConfiguration::paginate($modelBuilder);

        return $modelPath::getResource()::collectionWithTableSpecifications($products, $modelBuilder->getModel()->dataTableKey);
    }

    public function deleteListing(IntegrationInstance $integrationInstance, $id): Response
    {
        /** @var Model $modelPath */
        $modelPath = $integrationInstance->integration->getProductsModelPath();

        $builder = $modelPath::with('productListing');
        $builder->where('integration_instance_id', $integrationInstance->id);
        $builder->where($builder->getModel()->getKeyName(), e($id));
        $listing = $builder->firstOrFail();

        $reasons = $listing->delete();

        // check if the product is linked
        if ($reasons and is_array($reasons)) {
            foreach ($reasons as $key => $reason) {
                $this->response->addError($reason, ucfirst(Str::singular($key)).Response::CODE_RESOURCE_LINKED, $key, ['id' => $id]);
            }

            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->setMessage(__('messages.listing.delete_failed', ['id' => $id]));
        }

        return $this->response->setMessage(__('messages.success.delete', [
            'id' => $id,
            'resource' => 'listing',
        ]));
    }

    public function bulkDestroyListings(Request $request, IntegrationInstance $integrationInstance): Response
    {
        /** @var Model $modelPath */
        $modelPath = $integrationInstance->integration->getProductsModelPath();
        $modelInstance = new $modelPath();

        $request->validate([
            'filters' => 'required_without:ids',
            'ids' => 'required_without:filters|array|min:1',
            'ids.*' => "exists:{$this->getExistRuleTable($modelPath)}",
        ]);

        /** @var Builder $matching */
        $matching = $modelPath::with([])->where('integration_instance_id', $integrationInstance->id);
        $failedCount = 0;
        if ($request->has('ids')) {
            $ids = array_unique($request->input('ids', []));
            $matching = $matching->whereIn($modelInstance->getKeyName(), $ids)->get();
        } else {
            // Find by filters
            $matching = $matching->filter($request)->addRelations($request)->addCustomColumns($request)
                ->select($modelInstance->getKeyName())->get();
        }

        $total = $matching->count();
        foreach ($matching as $instance) {
            $failedCount += $instance->delete() === true ? 0 : 1;
        }

        if ($failedCount === $total) { // All failed
            return $this->response->setMessage(__('messages.failed.bulk_delete', ['resource' => 'listing']));
        }

        return $this->response->setMessage(__('messages.success.bulk_delete', [
            'total_count' => $total,
            'success_count' => ($total - $failedCount),
            'resource' => 'listing',
        ]));
    }

    public function isListingDeletable(Request $request, IntegrationInstance $integrationInstance): Response
    {
        /** @var Model $modelPath */
        $modelPath = $integrationInstance->integration->getProductsModelPath();
        $request->validate([
            'ids' => 'required|array|min:1',
            'ids.*' => "exists:{$this->getExistRuleTable($modelPath)}",
        ]);

        $ids = array_unique($request->input('ids', []));

        $result = [];
        $listings = $modelPath::with([])->whereIn((new $modelPath())->getKeyName(), $ids)->get();
        foreach ($listings as $key => $listing) {
            $isUsed = $listing->isUsed();
            $result[$key] = ['id' => $listing[$listing->getKeyName()]];
            $result[$key]['deletable'] = $isUsed ? false : true;
            $result[$key]['reason'] = $isUsed;
        }

        return $this->response->addData($result);
    }

    public function listingDocumentInSalesChannel(IntegrationInstance $integrationInstance, $input): Response
    {
        /** @var Model $modelPath */
        $modelPath = $integrationInstance->integration->getProductsModelPath();

        $builder = $modelPath::with('productListing')->where('integration_instance_id', $integrationInstance->id);
        if ($integrationInstance->isAmazonInstance()) {
            $builder->where(function ($builder) use ($input) {
                // TODO: Verify this works
                $builder->where('seller_sku', e($input));
            });
        } elseif ($integrationInstance->isShopify()) {
            $builder->where(function ($builder) use ($input) {
                $builder->where('variant_id', $input)->orWhere('sku', $input)->orWhere('id', e($input));
            });
        } else {
            $builder->where(function ($builder) use ($input) {
                $builder->where('sku', $input)->orWhere('_id', e($input));
            });
        }

        $modelBuilder = $builder->firstOrFail();

        return $this->response->addData($modelBuilder);
    }

    /**
     * @return RedirectResponse|BinaryFileResponse
     */
    public function exportListingsInSalesChannel(Request $request, IntegrationInstance $integrationInstance, $format)
    {
        /** @var Model|Listing $modelPath */
        $modelPath = $integrationInstance->integration->getProductsModelPath();

        $modelBuilder = $modelPath::with('productListing', 'productListing.product');

        $modelBuilder->filter($request);
        $modelBuilder->sort($request);
        $modelBuilder->options(['allowDiskUse' => true]);

        // to force apply on "and" conjunction
        $modelBuilder->where('integration_instance_id', $integrationInstance->id);

        if ($modelBuilder->count() <= 10000) {
            Telescope::stopRecording();

            $exportData = $modelPath::makeExportData($modelBuilder);
            $exported = ListingExporter::exportRecords($exportData, $format);

            return response()->download($exported);
        }
        // We download in a job
        $job = new DataTableExportJob(
            auth()->user(),
            class_basename($modelPath),
            $format,
            []
        );
        $job->setFilters($request->query('filters'));
        $job->setExtraQueryBindings([
            [
                'field' => 'integration_instance_id',
                'operator' => '=',
                'value' => $integrationInstance->id,
            ],
        ]);
        $job->setModel($modelPath);
        $job->setResource($modelPath::getResource());
        $job->setSorts($request->query('sortObjs'));
        if ($request->get('archived', 0) == 1) {
            $job->archived();
        }
        $job->setIncludedFields(json_decode($request->query('included'), true));

        dispatch($job)->onQueue('import-export');

        return back()->with(['message' => 'You will receive the data attached to an email shortly.']);
    }

    /**
     * Map product listing with channel listing, provided a Request
     * object.
     */
    public function mapListing(Request $request, IntegrationInstance $integrationInstance): Response
    {
        /** @var Model $modelPath */
        $modelPath = $integrationInstance->integration->getProductsModelPath();
        $collection = ( new $modelPath() )->getTable();

        /** @var ProductMapper $mapper */
        $mapper = $modelPath::getProductMapper($integrationInstance);

        $request->validate($mapper->validationRules());

        $mappings = $request->input('mapping', []);

        $mapResponse = $mapper->mapListings($mappings);

        // set the response message
        if (count($mapResponse['errors']) == count($mappings)) {
            $this->response->error(Response::HTTP_BAD_REQUEST)->setMessage(__('messages.failed.map_listings'));
        } elseif (count($mapResponse['errors'])) {
            $this->response->error(Response::HTTP_BAD_REQUEST)->setMessage(__('messages.integration_instance.some_listing_mappings_failed', ['failed' => count($mapResponse['errors'])]));
        } else {
            $this->response->addData(__('messages.success.create', ['resource' => 'product listings']));
        }
        // add map listing errors
        foreach ($mapResponse['errors'] as $error) {
            $this->response->addError(...$error);
        }

        return $this->response;
    }

    /**
     * Unmap product listing with channel listing.
     */
    public function unmapListing(Request $request, IntegrationInstance $integrationInstance): Response
    {
        /** @var Model $listingModelPath */
        $listingModelPath = $integrationInstance->integration->getProductsModelPath();

        $request->validate([
            'filters' => 'required_without:ids',
            'ids.*' => "required_without:filters|exists:{$this->getExistRuleTable($listingModelPath)}",
        ]);

        $ids = array_unique($request->input('ids', []));
        $data['totalCount'] = count($ids);
        $salesChannelProducts = $listingModelPath::with([])->when($request->has('ids'), function ($query) use ($ids) {
            $query->whereIn('_id', $ids);
        })->when(! $request->has('ids'), function ($query) use ($request) {
            $query->filter($request);
        });

        if ($request->has('filters')) {
            $data['totalCount'] = $salesChannelProducts->count();
        }

        DB::transaction(function () use ($listingModelPath, $salesChannelProducts) {
            // unmap
            $listingModelPath::whereIn('id', $salesChannelProducts->pluck('_id'))->update(['product' => null]);

            // Delete product listings
            ProductListing::with([])
                ->whereIn('document_id', $salesChannelProducts->pluck('_id'))
                ->where('document_type', $listingModelPath)
                ->delete();
            /** @see SKU-4215 it should only affect future ones I believe. */
            //            $mapper = $modelPath::getProductMapper($integrationInstance);
            //            $mapper->unmapListings($salesChannelProducts);
        });

        return $this->response->addData(__('messages.success.unmapped', [
            'resource' => 'product listings',
            'id' => '',
        ]));
    }

    /**
     * Lookup products
     *
     *
     * @return Response|AnonymousResourceCollection
     *
     * @deprecated {@see SalesChannelController::shippingMethods()
     */
    public function shippingServiceFromSalesChannel(Request $request, IntegrationInstance $integrationInstance)
    {
        /** @var Model $modelPath */
        $modelPath = $integrationInstance->integration->getShippingServiceModelPath();

        /** @var DataTableResource $resourcePath */
        $resourcePath = $integrationInstance->integration->getShippingServiceResourcePath();

        $modelBuilder = $modelPath::query();

        // only return "table_specifications" if its input is "1"
        if ($request->input('table_specifications') == 1) {
            return $this->response->addData(DataTableConfiguration::getTableSpecifications($modelBuilder->getModel()->dataTableKey));
        }

        $modelBuilder->filter($request);
        $modelBuilder->sort($request);

        // to force apply on "and" conjunction
        if ($integrationInstance->integration->name != Integration::NAME_AMAZON_US) {
            $modelBuilder->where('integration_instance_id', $integrationInstance->id);
        }

        // paginate with limit per page (default 10)
        $shippingService = DataTableConfiguration::paginate($modelBuilder);

        return $resourcePath::collectionWithTableSpecifications($shippingService, $modelBuilder->getModel()->dataTableKey);
    }

    /**
     * Map shipping services.
     *
     *
     * @deprecated {@see SalesChannelController::mapShippingMethods()
     */
    public function mapShippingMethods(Request $request, IntegrationInstance $integrationInstance): Response
    {
        /** @var Model $modelPath */
        $modelPath = $integrationInstance->integration->getShippingServiceModelPath();
        $collection = ( new $modelPath() )->getTable();

        $request->validate([
            'mapping' => 'required|array|min:1',
            'mapping.*.document_id' => "required|exists:{$collection},_id",
            'mapping.*.shipping_method_id' => 'nullable|exists:shipping_methods,id',
        ]);

        $mapping = $request->input('mapping', []);

        $salesChannelShippingMethods = $modelPath::with([])->whereIn('_id', array_column($mapping, 'document_id'))->get();

        foreach ($request->input('mapping') as $map) {
            $shippingMethodDocument = $salesChannelShippingMethods->firstWhere('_id', $map['document_id']);

            $integrationShippingMethod = IntegrationShippingMethod::with([])
                ->firstOrNew(
                    [
                        'integration_instance_id' => $integrationInstance->id,
                        'code' => $shippingMethodDocument->code,
                    ]
                );

            // sales_channel_listing_id based on integration instance
            $integrationShippingMethod->is_domestic = $shippingMethodDocument->domestic;
            $integrationShippingMethod->shipping_method_id = $map['shipping_method_id'];

            $status = $integrationShippingMethod->exists() ? MapShippingService::STATUS_UPDATED : MapShippingService::STATUS_NEW;
            if (isset($map['shipping_method_id'])) {
                $integrationShippingMethod->save();
            } elseif ($integrationInstance->exists) {
                $status = MapShippingService::STATUS_DELETED;
                $integrationShippingMethod->delete();
            }

            dispatch_sync(( new MapShippingService($integrationInstance, $shippingMethodDocument->name, $status) )->onConnection('sync'));
        }

        return $this->response->addData(__('messages.success.create', ['resource' => 'integration shipping methods']));
    }

    public function downloadOrders(Request $request, IntegrationInstance $integrationInstance)
    {
        if ($integrationInstance->isAmazonInstance()) {
            return $this->downloadMFNOrders($request, $integrationInstance);
        } elseif ($integrationInstance->isShopify()) {
            ShopifyGetOrdersJob::dispatch($integrationInstance, [
                ($request->date_method == 'created' ? 'created_after' : 'updated_after') => $request->from_date ?? null,
                ($request->date_method == 'created' ? 'created_before' : 'updated_before') => $request->to_date ?? null,
                'ids' => $request->ids,
            ]);

            return $this->response->setMessage('Added to the Queue, it will be processed shortly');
        } elseif ($integrationInstance->isMagento()) {
            $options = [];
            $status = $request->input('status');
            if ($status == 'open') {
                $options['status'] = ['processing'];
            } elseif ($status == 'closed') {
                $options['status'] = ['complete', 'closed', 'canceled'];
            }
            if ($date = $request->input('date')) {
                $options['createdAfter'] = $date;
            }
            dispatch(new \App\Jobs\Magento\GetOrdersJob($integrationInstance, $options))->onQueue('sales-channels');

            return $this->response->setMessage('Added to the Queue, it will be processed shortly');
        }

        return $this->response->addError(
            'Download Orders for '.$integrationInstance->integration->name.' is not implemented yet.',
            Response::CODE_UNACCEPTABLE,
            'DownloadOrders'
        );
    }

    public function getSalesChannelOrders($request, $integrationInstance)
    {
        /** @var Model $modelPath */
        $modelPath = $integrationInstance->integration->getOrderModelPath();
        $orders = $modelPath::with([]);

        // only return "table_specifications" if its input is "1"
        if ($request->input('table_specifications') == 1) {
            return $this->response->addData(DataTableConfiguration::getTableSpecifications($orders->getModel()->dataTableKey));
        }

        $orders->filter($request)
            ->sort($request)
            ->archived($request->input('archived', 0))
            ->where(function ($builder) use ($modelPath, $integrationInstance) {
                $builder->where('integration_instance_id', $integrationInstance->id)
                    ->whereNull((new $modelPath)->salesOrder()->getForeignKeyName()); // only unlinked with sku sales orders (orphans)
            });

        return $orders;
    }

    public function refreshWithSalesChannel(IntegrationInstance $integrationInstance, $orderId)
    {
        /** @var Model $modelPath */
        $modelPath = $integrationInstance->integration->getOrderModelPath();
        $order = $modelPath::with(['salesOrder'])->findOrFail($orderId);

        if (! ($order instanceof ShopifyOrder)) {
            return $this->response->addError('Order sync is only available for Shopify Orders', 'ShopifyOnly', $orderId);
        }

        try {
            $order = ShopifyDownloadOrder::refresh($order);

            return $this->response->addData(SalesOrderResource::make($order->salesOrder));
        } catch (\Exception $e) {
            return $this->response->error(Response::HTTP_INTERNAL_SERVER_ERROR, [$e->getMessage()])
                ->setMessage(__('messages.sales_order.sync_failed'));
        }
    }

    public function salesChannelOrders(Request $request, IntegrationInstance $integrationInstance)
    {
        $modelPath = $integrationInstance->integration->getOrderModelPath();

        // only return "table_specifications" if its input is "1"
        if ($request->input('table_specifications') == 1) {
            return $this->response->addData(DataTableConfiguration::getTableSpecifications((new $modelPath)->dataTableKey));
        }

        $orders = $this->getSalesChannelOrders($request, $integrationInstance);

        return $modelPath::getResource()::collectionWithTableSpecifications(DataTableConfiguration::paginate($orders), $orders->getModel()->dataTableKey);
    }

    public function salesChannelOrder(IntegrationInstance $integrationInstance, $orderId)
    {
        /** @var Model $modelPath */
        $modelPath = $integrationInstance->integration->getOrderModelPath();

        $order = $modelPath::with(['salesOrder'])->findOrFail($orderId);

        return $this->response->addData($order);
    }

    public function archiveSalesChannelOrder(IntegrationInstance $integrationInstance, $orderId)
    {
        /** @var Model $modelPath */
        $modelPath = $integrationInstance->integration->getOrderModelPath();

        $order = $modelPath::with([])->findOrFail($orderId);

        if ($order->archive()) {
            return $this->response
                ->setMessage(__('messages.success.archive', [
                    'resource' => 'order',
                    'id' => $order->getKeyName(),
                ]))
                ->addData(OrdersResource::make($order));
        }

        return $this->response
            ->addWarning(__('messages.failed.already_archive', [
                'resource' => 'order',
                'id' => $order->getKeyName(),
            ]), 'Order'.Response::CODE_ALREADY_ARCHIVED, 'id', ['id' => $order->getKeyName()])
            ->addData(OrdersResource::make($order));
    }

    public function bulkArchiveSalesChannelOrder(IntegrationInstance $integrationInstance, Request $request)
    {
        try {
            /** @var Model $modelPath */
            $modelPath = $integrationInstance->integration->getOrderModelPath();

            $request->validate([
                'filters' => 'required_without:ids',
                'ids' => 'required_without:filters|array|min:1',
            ]);

            $ids = array_unique($request->input('ids', []));
            $data['totalCount'] = count($ids);
            $data['failCount'] = 0;

            if ($request->has('ids')) {
                $orders = $modelPath::with([])->whereIn((new $modelPath)->getKeyName(), $ids)->get();
            } else {
                // Bulk selection with all rows was selected
                $request->merge(['archived' => 0]);
                $orders = $this->getSalesChannelOrders($request, $integrationInstance)->get();
            }

            foreach ($orders as $order) {
                if (! $order->archive()) {
                    $data['failCount']++;
                }
            }

            if ($data['failCount'] == $data['totalCount']) {
                $this->response->setMessage(__('messages.failed.bulk_archive', ['resource' => 'order']));
            }

            return $this->response->setMessage(__('messages.success.bulk_archive', [
                'total_count' => $data['totalCount'],
                'success_count' => ($data['totalCount'] - $data['failCount']),
                'resource' => 'order',
            ]));
        } catch (\Exception $e) {
            $this->response->setMessage(__('messages.failed.bulk_archive', ['resource' => 'order']));
        }
    }

    public function unarchiveSalesChannelOrder(IntegrationInstance $integrationInstance, $orderId)
    {
        /** @var Model $modelPath */
        $modelPath = $integrationInstance->integration->getOrderModelPath();

        $order = $modelPath::with([])->findOrFail($orderId);

        if ($order->unarchived()) {
            return $this->response
                ->setMessage(__('messages.success.unarchived', [
                    'resource' => 'order',
                    'id' => $order->getKeyName(),
                ]))
                ->addData(OrdersResource::make($order));
        }

        return $this->response
            ->addWarning(__('messages.failed.already_unarchived', [
                'resource' => 'order',
                'id' => $order->getKeyName(),
            ]), 'Order'.Response::CODE_ALREADY_UNARCHIVED, 'id', ['id' => $order->getKeyName()])
            ->addData(OrdersResource::make($order));
    }

    public function bulkUnarchiveSalesChannelOrder(IntegrationInstance $integrationInstance, Request $request)
    {
        /** @var Model $modelPath */
        $modelPath = $integrationInstance->integration->getOrderModelPath();

        $request->validate([
            'filters' => 'required_without:ids',
            'ids' => 'required_without:filters|array|min:1',
        ]);
        $ids = array_unique($request->input('ids', []));
        $data['totalCount'] = count($ids);
        $data['failCount'] = 0;

        if ($request->has('ids')) {
            $orders = $modelPath::with([])->whereIn((new $modelPath)->getKeyName(), $ids)->get();
        } else {
            $orders = $modelPath::with([])->filter($request)->archived(1)->get();
        }

        foreach ($orders as $order) {
            if (! $order->unarchived()) {
                $data['failCount'] = $data['failCount'] + 1;
            }
        }

        if ($data['failCount'] == $data['totalCount']) {
            return $this->response->setMessage(__('messages.failed.bulk_un_archive', ['resource' => 'order']));
        }

        return $this->response->setMessage(__('messages.success.bulk_un_archive', [
            'total_count' => $data['totalCount'],
            'success_count' => ($data['totalCount'] - $data['failCount']),
            'resource' => 'order',
        ]));
    }

    public function createSKUOrdersFromShopify(IntegrationInstance $integrationInstance, Request $request)
    {
        if (! $request->has('ids')) {
            $ids = ShopifyOrder::query()
                ->filter($request)
                ->pluck('id')
                ->toArray();
        } else {
            $ids = $request->input('ids', []);
        }

        dispatch(new GenerateCreateSalesOrderFromShopifyOrderJobJobs($integrationInstance, $ids))->onQueue('salesOrderProcessing');
    }

    public function CreateSkuOrderFromSalesChannelOrder(IntegrationInstance $integrationInstance, Request $request)
    {
        $request->validate([
            'filters' => 'required_without:ids',
            'ids' => 'required_without:filters|array|min:1',
        ]);

        if ($integrationInstance->isShopify()) {
            $this->createSKUOrdersFromShopify($integrationInstance, $request);

            return $this->response->setMessage('Added to the Queue, it will be process shortly');
        }

        /** @var Model $modelPath */
        $modelPath = $integrationInstance->integration->getOrderModelPath();

        $query = $modelPath::with([])
            ->when($request->has('ids'), function ($query) use ($request, $modelPath) {
                $query->whereIn((new $modelPath)->getKeyName(), array_unique($request->input('ids', [])));
            })
            ->when(! $request->has('ids'), function ($query) use ($request) {
                $query->filter($request);
            })
            ->whereNull((new $modelPath)->salesOrder()->getForeignKeyName());

        $data['totalCount'] = $query->count();

        if ($data['totalCount'] > 30) {
            dispatch(new BulkCreateSKUOrdersFromSalesChannelOrders($modelPath, $request->all()))->onQueue('sales-channels');

            return $this->response->setMessage('Added to the Queue, it will be process shortly');
        }

        $successCount = 0;
        $query->each(function ($order) use (&$successCount) {
            try {
                $order->createSKUOrder();
                $successCount++;
            } catch (\Throwable $exception) {
                $this->response->error(Response::HTTP_BAD_REQUEST)->addError($exception->getMessage(), $exception::class, 'id');
            }
        });

        return $this->response->setMessage(__('messages.success.bulk_create', [
            'total_count' => $data['totalCount'],
            'success_count' => $successCount,
            'resource' => 'order',
        ]));
    }

    public function previewSyncInventory(Request $request, IntegrationInstance $integrationInstance)
    {
        $locationId = $request->location_id;
        set_time_limit(0);
        $productListings = ProductListing::with(['product', 'salesChannel.integrationInstance.integration'])
            ->where('sales_channel_id', $integrationInstance->salesChannel->id)
            ->leftJoin('product_listing_inventory_locations', function ($join) {
                $join->on('product_listings.id', '=', 'product_listing_inventory_locations.product_listing_id');
            })
            ->whereNotNull('product_listing_inventory_locations.quantity', '>', 0)
            ->when(! empty($locationId), function ($query) use ($locationId) {
                $query->where('sales_channel_location_id', $locationId);
            })
            ->where('is_fba', false)
            ->whereNotNull('document_id') // document not deleted from database
            ->get();

        return PreviewSyncInventoryResource::collection(ProductListing::loadDocuments($productListings, $integrationInstance));
    }

    private function getExistRuleTable(string $modelPath): string
    {
        $modelInstance = new $modelPath();

        return "{$modelInstance->getTable()},{$modelInstance->getKeyName()}";
    }
}
