<?php

namespace App\Http\Controllers;

use App\DataTable\DataTable;
use App\Http\Controllers\Traits\BulkOperation;
use App\Http\Resources\IncotermResource;
use App\Models\Incoterm;
use App\Response;
use Exception;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class IncotermController extends Controller
{
    use bulkOperation, DataTable;

    private $resource = 'incoterm';

    protected $model_path = Incoterm::class;

    /**
     * Show an incoterm.
     */
    public function show(Incoterm $incoterm): Response
    {
        return $this->response->addData(IncotermResource::make($incoterm));
    }

    /**
     * Create a new incoterm.
     */
    public function store(Request $request): JsonResponse
    {
        $request->validate([
            'code' => 'required|max:255|unique:incoterms,code',
            'description' => 'required',
        ]);

        $incoterm = new Incoterm();
        $incoterm->code = $request->input('code');
        $incoterm->description = $request->input('description');
        $incoterm->save();

        return $this->response->success(Response::HTTP_CREATED)
            ->setMessage(__('messages.success.create', ['resource' => $this->resource]))
            ->addData(IncotermResource::make($incoterm));
    }

    /**
     * Update the incoterm.
     */
    public function update(Request $request, $id): JsonResponse
    {
        $request->validate([
            'code' => 'required|max:255|unique:incoterms,code,'.$id,
            'description' => 'required',
        ]);

        $incoterm = Incoterm::with([])->findOrFail($id);
        $incoterm->code = $request->input('code');
        $incoterm->description = $request->input('description');
        $incoterm->save();

        return $this->response
            ->setMessage(__('messages.success.update', [
                'resource' => $this->resource,
                'id' => $incoterm->code,
            ]))
            ->addData(IncotermResource::make($incoterm));
    }

    /**
     * Delete incoterm.
     *
     *
     * @throws Exception
     */
    public function destroy(Incoterm $incoterm): Response
    {
        $reasons = $incoterm->delete();

        // check if the nominalCode is linked
        if ($reasons and is_array($reasons)) {
            foreach ($reasons as $key => $reason) {
                $this->response->addError($reason, ucfirst(Str::singular($key)).Response::CODE_RESOURCE_LINKED, $key, ['incoterm_id' => $incoterm->id]);
            }

            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->setMessage(__('messages.failed.delete', [
                    'resource' => 'incoterm',
                    'id' => $incoterm->code,
                ]));
        }

        return $this->response->setMessage(__('messages.success.delete', [
            'resource' => 'incoterm',
            'id' => $incoterm->code,
        ]));
    }

    /**
     * Archive the incoterm.
     */
    public function archive(Incoterm $incoterm): JsonResponse
    {
        if ($incoterm->archive()) {
            return $this->response
                ->setMessage(__('messages.success.archive', [
                    'resource' => $this->resource,
                    'id' => $incoterm->code,
                ]))
                ->addData(IncotermResource::make($incoterm));
        }

        return $this->response
            ->addWarning(__('messages.failed.already_archive', [
                'resource' => $this->resource,
                'id' => $incoterm->code,
            ]), 'Incoterm'.Response::CODE_ALREADY_ARCHIVED, 'id', ['id' => $incoterm->id])
            ->addData(IncotermResource::make($incoterm));
    }

    /**
     * Unarchived the incoterm.
     */
    public function unarchived(Incoterm $incoterm): JsonResponse
    {
        if ($incoterm->unarchived()) {
            return $this->response
                ->setMessage(__('messages.success.unarchived', [
                    'resource' => $this->resource,
                    'id' => $incoterm->code,
                ]))
                ->addData(IncotermResource::make($incoterm));
        }

        return $this->response
            ->addWarning(__('messages.failed.unarchived', [
                'resource' => 'incoterm',
                'id' => $incoterm->code,
            ]), 'Incoterm'.Response::CODE_ALREADY_UNARCHIVED, 'id', ['id' => $incoterm->id])
            ->addData(IncotermResource::make($incoterm));
    }

    /**
     * bulk delete using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkDestroy(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_DELETE);
    }

    /**
     * bulk archive using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_ARCHIVE);
    }

    /**
     * bulk un archive using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkUnArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_UN_ARCHIVE);
    }

    /**
     * check the possibility of deletion.
     */
    public function isDeletable(Request $request): Response
    {
        // validate
        $request->validate([
            'ids' => 'required|array|min:1',
            'ids.*' => 'integer|exists:incoterms,id',
        ]);

        $ids = array_unique($request->input('ids', []));

        $result = [];
        $incoterms = Incoterm::with([])->whereIn('id', $ids)->select('id', 'code')->get();
        foreach ($incoterms as $key => $incoterm) {
            $isUsed = $incoterm->isUsed();

            $result[$key] = $incoterm->only('id', 'code');
            $result[$key]['deletable'] = ! boolval($isUsed);
            $result[$key]['reason'] = $isUsed ?: null;
        }

        return $this->response->addData($result);
    }

    /**
     * {@inheritDoc}
     */
    protected function getModel()
    {
        return Incoterm::class;
    }

    /**
     * {@inheritDoc}
     */
    protected function getResource()
    {
        return IncotermResource::class;
    }
}
