<?php

namespace App\Http\Controllers;

use App\Http\Resources\Financials\TopProductResource;
use App\Managers\ReportManager;
use App\Models\Product;
use App\Repositories\SalesOrderLineFinancialsRepository;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\AnonymousResourceCollection;
use Illuminate\Http\Resources\Json\JsonResource;
use Illuminate\Validation\Rule;

class FinancialController extends Controller
{
    private SalesOrderLineFinancialsRepository $salesOrderLineFinancialsRepository;

    private ReportManager $reportManager;

    public function __construct()
    {
        parent::__construct();
        $this->salesOrderLineFinancialsRepository = app(SalesOrderLineFinancialsRepository::class);
        $this->reportManager = app(ReportManager::class);
    }

    /**
     * @throws Exception
     */
    public function summaryByPeriod(Request $request): AnonymousResourceCollection
    {
        $request->validate([
            'period' => 'required|string',
            'trailing_days' => 'nullable|integer',
        ]);

        return JsonResource::collection(
            $this->salesOrderLineFinancialsRepository->getSummaryByPeriod($request->input('period'), $request->input('trailing_days'))
        )
            ->additional([
                'status' => __('messages.status_success'),
            ]);
    }

    public function summaryByProductPeriod(Request $request, int $product_id)
    {
        $request->validate([
            'period' => 'required|string',
            'trailing_days' => 'nullable|integer',
        ]);

        return JsonResource::collection(
            $this->salesOrderLineFinancialsRepository->getSummaryByProductPeriod(Product::findOrFail($product_id), $request->input('period'), $request->input('trailing_days'))
        )
            ->additional([
                'status' => __('messages.status_success'),
            ]);
    }

    public function topProducts(Request $request)
    {
        $request->validate([
            'metric' => [
                'required',
                Rule::in('revenue', 'profit'),
            ],
            'limit' => 'nullable|integer|gt:0',
        ]);
        $metric = $request->input('metric', 'revenue');
        $limit = $request->input('limit', 10);

        return TopProductResource::collection($this->reportManager->topProducts($metric, $limit))
            ->additional([
                'status' => __('messages.status_success'),
            ]);
    }
}
