<?php

namespace App\Http\Controllers\EBay;

use App\Http\Controllers\Controller;
use App\Integrations\Ebay;
use App\Models\Integration;
use App\Models\IntegrationInstance;
use App\Models\SalesChannelsStateRedis;
use Exception;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class EBayIntegrationController extends Controller
{
    protected $credentials;

    public function __construct()
    {
        parent::__construct();

        $integration = Integration::with([])->where('name', Integration::NAME_EBAY)->first();

        if ($integration) {
            $this->credentials = $integration->credentials;
        }
    }

    /**
     * Generate ebay oauth user url with state to know which user authorized.
     */
    public function generateOAuthUrl(IntegrationInstance $integrationInstance): JsonResponse
    {
        $generate_result = ( new Ebay($this->credentials) )->generateOAuthUrl();

        SalesChannelsStateRedis::updateOrCreate(
            $generate_result['state'],
            [
                'type' => Integration::NAME_EBAY,
                'type_id' => $integrationInstance->id,
            ]
        );

        return response()->json($generate_result);
    }

    /**
     * Save ebay user code.
     */
    public function ebayAuthAccepted(Request $request)
    {
        $request->validate(
            [
                'state' => 'required',
                'code' => 'required',
            ]
        );

        //    Get state from Redis to store user code.
        $state = SalesChannelsStateRedis::find($request->get('state'));
        if (! $state || $state['type'] != Integration::NAME_EBAY) {
            return response()->json(
                [
                    'status' => 'failed',
                    'message' => 'State not exists.',
                ]
            );
        }

        $integrationInstance = IntegrationInstance::with([])->findOrFail($state['type_id']);

        if (! $integrationInstance) {
            return response()->json(
                [
                    'status' => 'failed',
                    'message' => 'Ebay sales channel not exists on DB.',
                ]
            );
        }
        try {
            $access_token = ( new Ebay($this->credentials) )->getAccessToken($request->get('code'));
        } catch (Exception $exception) {
            return response()->json(
                [
                    'status' => 'failed',
                    'message' => $exception->getMessage(),
                ]
            );
        }
        /**
         * Save Data on Redis.
         */
        $state['code'] = $request->get('code');
        $state['access_token'] = $access_token['access_token'];
        $state['token_type'] = $access_token['token_type'];
        $state['expires_in'] = $access_token['expires_in'];
        $state['refresh_token'] = $access_token['refresh_token'];

        $integrationInstance->connection_settings = $state;
        $integrationInstance->save();

        return redirect()->to('/');
    }
}
