<?php

namespace App\Http\Controllers;

use App\DataTable\DataTable;
use App\Http\Controllers\Traits\BulkOperation;
use App\Http\Requests\DataFeed\IsDeletableRequest;
use App\Http\Requests\StoreDataFeedRequest;
use App\Http\Resources\DataFeedResource;
use App\Jobs\DataFeed\ImportFileDataFeed;
use App\Models\DataFeed;
use App\Response;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Str;

class DataFeedController extends Controller
{
    use BulkOperation, DataTable;

    protected $model_path = DataFeed::class;

    protected $resource = 'data feed';

    protected function getModel(): string
    {
        return DataFeed::class;
    }

    protected function getResource(): string
    {
        return DataFeedResource::class;
    }

    /**
     * Data Feed.
     *
     *
     * @throws Throwable
     */
    public function store(StoreDataFeedRequest $request): Response
    {
        $dataFeed = new DataFeed($request->all());
        $dataFeed->save();

        //Raise Job for Data Feed
        ImportFileDataFeed::dispatch($dataFeed->id);

        return $this->response->success(Response::HTTP_CREATED)
            ->setMessage(__('messages.success.create', ['resource' => $this->resource]))
            ->addData(DataFeedResource::make($dataFeed));
    }

    /**
     * Update a DataFeed.
     *
     *
     * @throws Throwable
     */
    public function update(StoreDataFeedRequest $request, $id): Response
    {
        $dataFeed = DataFeed::findOrFail($id);
        $dataFeed->fill($request->all());
        $dataFeed->update();

        return $this->response
            ->setMessage(__('messages.success.update', [
                'resource' => $this->resource,
                'id' => $dataFeed->name,
            ]))
            ->addData(DataFeedResource::make($dataFeed));
    }

    public function show(DataFeed $dataFeed): Response
    {
        return $this->response->addData(DataFeedResource::make($dataFeed));
    }

    /**
     * @param  DataFeed  $dataFeed
     *
     * @throws Exception
     */
    public function destroy($id): Response
    {
        $dataFeed = DataFeed::findOrFail($id);
        $reasons = $dataFeed->delete();

        // check if the datafeed is linked
        if ($reasons and is_array($reasons)) {
            foreach ($reasons as $key => $reason) {
                $this->response->addError($reason, ucfirst(Str::singular($key)).Response::CODE_RESOURCE_LINKED, $key, ['product_id' => $id]);
            }

            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->setMessage(__('messages.data_feed.delete_failed', ['id' => $dataFeed->id]));
        }

        return $this->response->setMessage(__('messages.success.delete', [
            'resource' => $this->resource,
            'id' => $dataFeed->id,
        ]));
    }

    public function isDeletable(IsDeletableRequest $request)
    {
        $records = DataFeed::whereIn('id', $request->ids)
            ->select('id', 'name')
            ->get();

        return $this->response->addData($records);
    }

    public function archive($id)
    {
        $dataFeed = DataFeed::findOrFail($id);

        if ($dataFeed->archive()) {
            return $this->response->setMessage(__('messages.success.archive', [
                'resource' => $this->resource,
                'id' => $dataFeed->id,
            ]));
        }

        return $this->response->addWarning(__('messages.failed.already_archive', [
            'resource' => $this->resource,
            'id' => $dataFeed->id,
        ]), 'DataFeed'.Response::CODE_ALREADY_ARCHIVED, 'id', ['id' => $id]);
    }

    public function unarchived($id)
    {
        $dataFeed = DataFeed::findOrFail($id);

        if ($dataFeed->unarchived()) {
            return $this->response->setMessage(__('messages.success.unarchived', [
                'resource' => $this->resource,
                'id' => $dataFeed->id,
            ]));
        }

        return $this->response->warning()
            ->addWarning(__('messages.failed.unarchived', [
                'resource' => 'product',
                'id' => $dataFeed->id,
            ]), 'DataFeed'.Response::CODE_ALREADY_UNARCHIVED, 'id', ['id' => $id]);
    }

    public function bulkArchive(Request $request)
    {
        return $this->bulkOperation($request, $this->BULK_ARCHIVE);
    }

    public function bulkUnarchive(Request $request)
    {
        return $this->bulkOperation($request, $this->BULK_UN_ARCHIVE);
    }

    public function importConfig($dataFeed)
    {
        return Config::get('data_feed.'.$dataFeed);
    }

    public function previewData()
    {
    }

    /**
     * Bulk delete nominal codes.
     *
     *
     * @throws Exception
     */
    public function bulkDestroy(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_DELETE);
    }
}
