<?php

namespace App\Http\Controllers;

use App\DataTable\DataTableConfiguration;
use App\Helpers;
use App\Http\Resources\ReportingProductFinancials;
use App\Models\ReportingDailyFinancial;
use App\Models\ReportingDailyProductAccountingFinancials;
use App\Models\SalesOrderLine;
use App\Models\Setting;
use Illuminate\Contracts\Pagination\Paginator;
use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\AnonymousResourceCollection;
use Illuminate\Http\Resources\Json\JsonResource;
use Illuminate\Support\Facades\DB;
use Spatie\QueryBuilder\AllowedFilter;
use Spatie\QueryBuilder\QueryBuilder;

class DashboardController extends Controller
{
    private $dateFields = [
        'order_date',
        'reporting_date',
    ];

    /**
     * Query on reporting products.
     */
    public function products(Request $request): Paginator
    {
        $reportingProducts = SalesOrderLine::with(['product', 'salesOrder'])
            ->join('sales_order_line_financials', 'sales_order_lines.id', '=', 'sales_order_line_financials.sales_order_line_id')
            ->filter();

        if ($request->filled('group_by')) {
            $reportingProducts->groupBy($request->query('group_by'));

            $select = $this->bindGrouping($request);
            $select .= 'SUM(`quantity`) as quantity_sold,';
            $select .= 'COUNT(DISTINCT sales_order_id) as num_orders';
            $select .= 'SUM(`quantity` * (`amount` - IF(`is_tax_included` = 1, `tax_allocation`, 0)) + sales_order_line_financials.revenue_allocated - sales_order_line_financials.credits) as revenue,';
            $select .= 'SUM(sales_order_line_financials.`cogs` + 
                        sales_order_line_financials.`cost_allocated` - 
                        sales_order_line_financials.`cogs_returned`) as cost_order';

            $reportingProducts->select(DB::raw($select));
            $reportingProducts->sort();
        }

        return DataTableConfiguration::paginate($reportingProducts);
    }

    /**
     * Query on reporting orders.
     */
    public function orders(Request $request): AnonymousResourceCollection
    {
        /*
         * TODO: Move database portion to repository
         */
        return JsonResource::collection(
            SalesOrderLine::query()
                ->filter() // does not seem to be working... needs to interpret filters from payload
                ->selectRaw("DATE_FORMAT(CONVERT_TZ(sales_orders.order_date, 'UTC', '".Helpers::setting(Setting::KEY_DEFAULT_TIMEZONE)."'), '%Y-%m-%d') as order_date")
                ->selectRaw('SUM(sales_order_line_financials.total_revenue)')
                ->selectRaw('SUM(sales_order_line_financials.total_cost)')
                ->join('sales_order_line_financials', 'sales_order_line_financials.sales_order_line_id', 'sales_order_lines.id')
                ->join('sales_orders', 'sales_orders.id', 'sales_order_lines.sales_order_id')
                ->groupBy(DB::raw("DATE_FORMAT(CONVERT_TZ(sales_orders.order_date, 'UTC', '".Helpers::setting(Setting::KEY_DEFAULT_TIMEZONE)."'), '%Y-%m-%d')"))
                ->get()
        )
            ->additional([
                'status' => __('messages.status_success'),
            ]);
    }

    /**
     * Gets the top-selling products with their corresponding quantities sold.
     * Use filters to narrow down the products.
     */
    public function topProducts($metric = 'revenue')
    {
        /*
         * TODO: Move database portion to repository
         */

        $select = 'reporting_daily_financials.product_id, ';
        $select .= 'SUM(`quantity`) as quantity_sold, ';
        $select .= 'SUM(`total_revenue`) as total_revenue, ';
        $select .= 'SUM(`total_cost`) as total_cost, ';
        $select .= 'SUM(`total_profit`) as profit, ';
        $select .= 'SUM(`num_orders`) as num_orders';
        $query = QueryBuilder::for(ReportingDailyFinancial::class)
            ->join('products', 'products.id', 'reporting_daily_financials.product_id')
            ->join('product_brands', 'product_brands.id', 'products.brand_id')
            ->groupBy(['products.id'])
            ->orderBy($metric, 'desc')
            ->select(DB::raw($select))
            ->allowedFilters([
                AllowedFilter::scope('start_date'),
                AllowedFilter::scope('end_date'),
            ]);

        return $this->response->addData(ReportingProductFinancials::collectionWithTableSpecifications($query, ReportingDailyProductAccountingFinancials::class));
    }

    public function mostProfitableProducts()
    {
        return $this->topProducts('profit');
    }

    private function isDateField($field): bool
    {
        return in_array($field, $this->dateFields);
    }

    /**
     * @return mixed
     */
    private function bindGrouping(Request $request)
    {
        $groupBy = $request->input('group_by');
        if (! $groupBy) {
            return '';
        }

        if ($this->isDateField($groupBy)) {
            return "DATE_FORMAT({$groupBy}, '%Y-%m-%d') as {$groupBy},";
        } else {
            return $groupBy.',';
        }
    }
}
