<?php

namespace App\Http\Controllers;

use App\DataTable\DataTable;
use App\Http\Controllers\Traits\BulkOperation;
use App\Http\Controllers\Traits\ImportsData;
use App\Http\Requests\StoreAddressRequest;
use App\Http\Requests\StoreCustomer;
use App\Http\Resources\CustomerAddressesResource;
use App\Http\Resources\CustomerResource;
use App\Models\Address;
use App\Models\Customer;
use App\Models\SalesChannel;
use App\Response;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\AnonymousResourceCollection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Throwable;

class CustomerController extends Controller
{
    use BulkOperation, DataTable, ImportsData;

    protected $model_path = Customer::class;

    public function show(Customer $customer)
    {
        $customer->load([
            'addresses',
            'billingAddress',
            'shippingAddress',
            'salesChannelOrigin',
            'productPricingTier',
            'store',
            'shippingMethodDomestic',
            'shippingMethodInternational',
            'warehouse',
        ]);

        return $this->response->addData(CustomerResource::make($customer));
    }

    /**
     * Create a new customer.
     *
     *
     * @throws Throwable
     */
    public function store(StoreCustomer $request): Response
    {
        $inputs = $request->validated();

        DB::transaction(function () use ($inputs) {
            $address = new Address($inputs);
            $address->label = empty($inputs['label']) ? 'Shipping Address' : $inputs['label'];

            $customer = Customer::exists($inputs);

            if ($customer) {
                $this->response->error(422)
                    ->setMessage(__('messages.customer.customer_exists'))
                    ->addError(__('messages.customer.customer_exists'), Response::CODE_CUSTOMER_EXISTS, 'address', [
                        'address' => $inputs,
                        'existing_customer_id' => $customer->id,
                    ]);
            } else {
                $address->save();

                $customer = new Customer();
                $customer->sales_channel_origin_id = SalesChannel::LOCAL_CHANNEL_ID;

                if (! empty($inputs['default_billing_address'])) {
                    $billingAddress = new Address($inputs['default_billing_address']);
                    $billingAddress->label = empty($inputs['default_billing_address']['label']) ? 'Billing Address' : $inputs['default_billing_address']['label'];
                    $billingAddress->save();

                    $customer->default_billing_address_id = $billingAddress->id;
                } else {
                    $address->label = 'Default Address';
                    $address->save();

                    $customer->default_billing_address_id = $address->id;
                }

                $customer->default_shipping_address_id = $address->id;
                $customer->fill($inputs);
                $customer->save();

                $customer->load('addresses', 'billingAddress', 'shippingAddress');

                $this->response->setMessage(__('messages.success.create', ['resource' => 'customer']))
                    ->addData(CustomerResource::make($customer));
            }
        });

        return $this->response;
    }

    /**
     * Update a customer.
     *
     *
     * @throws Throwable
     */
    public function update(StoreCustomer $request, $id): Response
    {
        $inputs = $request->validated();

        DB::transaction(function () use ($id, $inputs) {
            $customer = Customer::with(['address'])->findOrFail($id);

            $existedCustomer = Customer::exists($inputs);

            if ($existedCustomer and $existedCustomer->id != $id) {
                $this->response->error(422)
                    ->setMessage(__('messages.customer.customer_exists'))
                    ->addError(__('messages.customer.customer_exists'), Response::CODE_CUSTOMER_EXISTS, 'address', [
                        'address' => $inputs,
                        'existing_customer_id' => $customer->id,
                    ]);
            } else {
                $customer->fill($inputs);
                $customer->save();

                $customer->load([
                    'addresses',
                    'billingAddress',
                    'shippingAddress',
                    'salesChannelOrigin',
                    'productPricingTier',
                    'store',
                    'shippingMethodDomestic',
                    'shippingMethodInternational',
                    'warehouse',
                ]);

                $this->response
                    ->setMessage(__('messages.success.update', [
                        'resource' => 'customer',
                        'id' => $customer->name,
                    ]))
                    ->addData(CustomerResource::make($customer));
            }
        });

        return $this->response;
    }

    /**
     * Find customer by depending on name, zip and 90% match of address1.
     */
    public function findMatch(Request $request): AnonymousResourceCollection
    {
        $request->validate(
            [
                'name' => 'required|string',
                'zip' => 'required|string',
                'address1' => 'required|string',
            ]
        );

        return CustomerResource::collection(Customer::findMatch($request))->additional(['status' => __('messages.status_success')]);
    }

    /**
     * Archive the customer.
     */
    public function archive(Customer $customer): Response
    {
        if ($customer->archive()) {
            return $this->response
                ->setMessage(__('messages.success.archive', [
                    'resource' => 'customer',
                    'id' => $customer->name,
                ]))
                ->addData(CustomerResource::make($customer));
        }

        return $this->response
            ->addWarning(__('messages.failed.already_archive', [
                'resource' => 'customer',
                'id' => $customer->name,
            ]), 'Customer'.Response::CODE_ALREADY_ARCHIVED, 'id', ['id' => $customer->id])
            ->addData(CustomerResource::make($customer));
    }

    /**
     * Unarchived the customer.
     */
    public function unarchived(Customer $customer): Response
    {
        if ($customer->unarchived()) {
            return $this->response
                ->setMessage(__('messages.success.unarchived', [
                    'resource' => 'customer',
                    'id' => $customer->name,
                ]))
                ->addData(CustomerResource::make($customer));
        }

        return $this->response
            ->addWarning(__('messages.failed.unarchived', [
                'resource' => 'customer',
                'id' => $customer->name,
            ]), 'Customer'.Response::CODE_ALREADY_UNARCHIVED, 'id', ['id' => $customer->id])
            ->addData(CustomerResource::make($customer));
    }

    /**
     * Delete customer.
     *
     *
     * @throws Exception
     */
    public function destroy(Customer $customer): Response
    {
        $reasons = $customer->delete();

        // check if the customer is linked
        if ($reasons and is_array($reasons)) {
            foreach ($reasons as $key => $reason) {
                $this->response->addError($reason, ucfirst(Str::singular($key)).Response::CODE_RESOURCE_LINKED, $key, ['customer_id' => $customer->id]);
            }

            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->setMessage(__('messages.failed.delete', [
                    'resource' => 'customer',
                    'id' => $customer->name,
                ]));
        }

        return $this->response->setMessage(__('messages.success.delete', [
            'resource' => 'customer',
            'id' => $customer->name,
        ]));
    }

    /**
     * Bulk delete nominal codes.
     *
     *
     * @throws Exception
     */
    public function bulkDestroy(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_DELETE);
    }

    /**
     * Bulk archive nominal codes.
     */
    public function bulkArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_ARCHIVE);
    }

    /**
     * Bulk unarchived nominal codes.
     *
     *
     * @return mixed
     */
    public function bulkUnArchive(Request $request)
    {
        return $this->bulkOperation($request, $this->BULK_UN_ARCHIVE);
    }

    /**
     * check the possibility of deletion.
     */
    public function isDeletable(Request $request): Response
    {
        // validate
        $request->validate([
            'ids' => 'required|array|min:1',
            'ids.*' => 'integer|exists:customers,id',
        ]);

        $ids = array_unique($request->input('ids', []));

        $result = [];
        $customers = Customer::with([])->whereIn('id', $ids)->select('id', 'name')->get();
        foreach ($customers as $key => $customer) {
            $isUsed = $customer->isUsed();

            $result[$key] = $customer->only('id', 'name');
            $result[$key]['deletable'] = ! boolval($isUsed);
            $result[$key]['reason'] = $isUsed ?: null;
        }

        return $this->response->addData($result);
    }

    /**
     * Retrieve all addresses of a particular customer.
     */
    public function addresses(Customer $customer): Response
    {
        $customer->load('addresses', 'billingAddress', 'shippingAddress');

        return $this->response->addData(CustomerAddressesResource::make($customer));
    }

    /**
     * add other address to customer.
     */
    public function addAddress(StoreAddressRequest $request, $id): Response
    {
        $inputs = $request->validated();

        $customer = Customer::with([])->findOrFail($id);

        $address = new Address($inputs);
        $address->save();

        $isDefaultShipping = $inputs['is_default_shipping'] ?? false;
        $isDefaultBilling = $inputs['is_default_billing'] ?? false;

        // set as default shipping address
        if ($isDefaultShipping) {
            $customer->setDefaultShippingAddress($address->id);
        }
        // set as default billing address
        if ($isDefaultBilling) {
            $customer->setDefaultBillingAddress($address->id);
        }
        // add to other addresses
        if (! $isDefaultShipping && ! $isDefaultBilling) {
            $customer->addresses()->attach($address);
        }

        $customer->load('addresses', 'billingAddress', 'shippingAddress');

        return $this->response->addData(CustomerAddressesResource::make($customer));
    }

    /**
     * Set default shipping address.
     */
    public function setDefaultShippingAddress(Customer $customer, $addressId): Response
    {
        $asDefaultShipping = true; // request()->input( 'check', false );

        $customer->setDefaultShippingAddress($addressId, $asDefaultShipping);

        $customer->load('addresses', 'billingAddress', 'shippingAddress');

        return $this->response->setMessage(__('messages.customer.'.($asDefaultShipping ? 'set_default_address' : 'unset_default_address'), ['type' => 'shipping']))
            ->addData(CustomerAddressesResource::make($customer));
    }

    /**
     * Set default billing address.
     */
    public function setDefaultBillingAddress(Customer $customer, $addressId): Response
    {
        $asDefaultBilling = true; // request()->input( 'check', false );

        $customer->setDefaultBillingAddress($addressId, $asDefaultBilling);

        $customer->load('addresses', 'billingAddress', 'shippingAddress');

        return $this->response->setMessage(__('messages.customer.'.($asDefaultBilling ? 'set_default_address' : 'unset_default_address'), ['type' => 'billing']))
            ->addData(CustomerAddressesResource::make($customer));
    }

    protected function getModel()
    {
        return Customer::class;
    }

    protected function getResource()
    {
        return CustomerResource::class;
    }
}
