<?php

namespace App\Http\Controllers;

use App\DataTable\DataTable;
use App\Http\Controllers\Traits\BulkOperation;
use App\Http\Controllers\Traits\ImportsData;
use App\Http\Requests\StoreAttributeGroup;
use App\Http\Resources\AttributeGroupDetailsResource;
use App\Http\Resources\AttributeGroupResource;
use App\Models\Attribute;
use App\Models\AttributeGroup;
use App\Response;
use Exception;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\AnonymousResourceCollection;
use Illuminate\Support\Str;

class AttributeGroupController extends Controller
{
    use BulkOperation, DataTable, ImportsData;

    protected $model_path = AttributeGroup::class;

    /**
     * Resource Name.
     *
     * @var string
     */
    private $resource = 'attribute group';

    /**
     * Display a listing of the attribute group.
     */
    public function indexForManage(): AnonymousResourceCollection
    {
        $attributeGroups = AttributeGroup::with(['productsIdsThroughCategories', 'attributes'])
            ->withCount(['categories', 'products'])
            ->orderBy('sort_order', 'ASC')
            ->get();

        return AttributeGroupResource::collection($attributeGroups)->additional(['status' => __('messages.status_success')]);
    }

    /**
     * Display the specified attribute group.
     */
    public function show(int $id): Response
    {
        $attributeGroup = AttributeGroup::with([
            'productsIdsThroughCategories',
            'attributes',
            'categories',
            'products',
        ])
            ->findOrFail($id);

        return $this->response->addData(AttributeGroupDetailsResource::make($attributeGroup));
    }

    /**
     * Store a newly created attribute group.
     */
    public function store(StoreAttributeGroup $request): JsonResponse
    {
        $inputs = $request->validated();

        $attributeGroup = new AttributeGroup($inputs);
        $attributeGroup->save();

        // store its attributes
        if (! empty($inputs['attributes'])) {
            foreach ($inputs['attributes'] as $attributeInput) {
                $attribute = Attribute::with([])->findOrFail($attributeInput['id']);
                $attribute->attribute_group_id = $attributeGroup->id;

                if (! empty($attributeInput['sort_order'])) {
                    $attribute->sort_order = $attributeInput['sort_order'];
                }

                $attribute->save();
            }
        }

        $attributeGroup->load('attributes');

        return $this->response->success(Response::HTTP_CREATED, __('messages.success.create', ['resource' => $this->resource]))
            ->addData(AttributeGroupDetailsResource::make($attributeGroup));
    }

    /**
     * Update the specified attribute group.
     */
    public function update(StoreAttributeGroup $request, int $id): JsonResponse
    {
        $inputs = $request->validated();

        $attributeGroup = AttributeGroup::with(['attributes'])->findOrFail($id);
        $attributeGroup->fill($inputs);
        $attributeGroup->save();

        // sync its attributes
        if (array_key_exists('attributes', $inputs)) {
            foreach ($inputs['attributes'] as $attributeInput) {
                $attribute = Attribute::with([])->findOrFail($attributeInput['id']);
                $attribute->attribute_group_id = $attributeGroup->id;

                if (! empty($attributeInput['sort_order'])) {
                    $attribute->sort_order = $attributeInput['sort_order'];
                }

                $attribute->save();
            }

            $attributeIds = array_column($inputs['attributes'] ?: [], 'id');
            Attribute::with([])->where('attribute_group_id', $attributeGroup->id)->whereNotIn('id', $attributeIds)->update(['attribute_group_id' => null]);
        }

        $attributeGroup->load('attributes');

        return $this->response
            ->setMessage(__('messages.success.update', [
                'resource' => $this->resource,
                'id' => $attributeGroup->name,
            ]))
            ->addData(AttributeGroupDetailsResource::make($attributeGroup));
    }

    /**
     * Remove the specified attribute group.
     *
     *
     * @throws Exception
     */
    public function destroy(int $id): JsonResponse
    {
        $attributeGroup = AttributeGroup::with([])->findOrFail($id);

        $reasons = $attributeGroup->delete();

        // check if the nominalCode is linked
        if ($reasons and is_array($reasons)) {
            foreach ($reasons as $key => $reason) {
                $this->response->addError($reason, ucfirst(Str::singular($key)).Response::CODE_RESOURCE_LINKED, $key, ['attribute_group_id' => $attributeGroup->id]);
            }

            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->setMessage(__('messages.failed.delete', [
                    'resource' => $this->resource,
                    'id' => $attributeGroup->name,
                ]));
        }

        return $this->response->setMessage(__('messages.success.delete', [
            'resource' => $this->resource,
            'id' => $attributeGroup->name,
        ]));
    }

    /**
     * Archive the attribute group.
     */
    public function archive(AttributeGroup $attributeGroup): Response
    {
        if ($attributeGroup->archive()) {
            return $this->response
                ->setMessage(__('messages.success.archive', [
                    'resource' => $this->resource,
                    'id' => $attributeGroup->name,
                ]))
                ->addData(AttributeGroupResource::make($attributeGroup));
        }

        return $this->response->warning()
            ->addWarning(__('messages.failed.already_archive', [
                'resource' => $this->resource,
                'id' => $attributeGroup->name,
            ]), 'AttributeGroup'.Response::CODE_ALREADY_ARCHIVED, 'id', ['id' => $attributeGroup->id])
            ->addData(AttributeGroupResource::make($attributeGroup));
    }

    /**
     * Unarchived the attribute group.
     */
    public function unarchived(AttributeGroup $attributeGroup): Response
    {
        if ($attributeGroup->unarchived()) {
            return $this->response
                ->setMessage(__('messages.success.unarchived', [
                    'resource' => $this->resource,
                    'id' => $attributeGroup->name,
                ]))
                ->addData(AttributeGroupResource::make($attributeGroup));
        }

        return $this->response->warning()
            ->addWarning(__('messages.failed.unarchived', [
                'resource' => $this->resource,
                'id' => $attributeGroup->name,
            ]), 'AttributeGroup'.Response::CODE_ALREADY_UNARCHIVED, 'id', ['id' => $attributeGroup->id])
            ->addData(AttributeGroupResource::make($attributeGroup));
    }

    /**
     * bulk archive using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_ARCHIVE);
    }

    /**
     * bulk un archive using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkUnArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_UN_ARCHIVE);
    }

    /**
     * Bulk delete attribute groups.
     *
     *
     * @throws Exception
     */
    public function bulkDestroy(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_DELETE);
    }

    /**
     * check the possibility of deletion.
     */
    public function isDeletable(Request $request): Response
    {
        // validate
        $request->validate([
            'ids' => 'required|array|min:1',
            'ids.*' => 'integer|exists:attribute_groups,id',
        ]);

        $ids = array_unique($request->input('ids', []));

        $result = [];
        $attributeGroups = AttributeGroup::with([])->whereIn('id', $ids)->select('id', 'name')->get();
        foreach ($attributeGroups as $key => $attributeGroup) {
            $isUsed = $attributeGroup->isUsed();

            $result[$key] = $attributeGroup->only('id', 'name');
            $result[$key]['deletable'] = ! boolval($isUsed);
            $result[$key]['reason'] = $isUsed ?: null;
        }

        return $this->response->addData($result);
    }

    /**
     * {@inheritDoc}
     */
    protected function getModel()
    {
        return AttributeGroup::class;
    }

    /**
     * {@inheritDoc}
     */
    protected function getResource()
    {
        return AttributeGroupResource::class;
    }
}
