<?php

namespace App\Http\Controllers;

use App\DataTable\DataTable;
use App\Http\Controllers\Traits\BulkOperation;
use App\Http\Controllers\Traits\ImportsData;
use App\Http\Requests\StoreAttribute;
use App\Http\Resources\AttributeGroupResource;
use App\Http\Resources\AttributeResource;
use App\Models\Attribute;
use App\Models\AttributeGroup;
use App\Response;
use Exception;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class AttributeController extends Controller
{
    use BulkOperation, DataTable, ImportsData;

    protected $model_path = Attribute::class;

    private $resource = 'attribute';

    /**
     * Show the attribute.
     */
    public function show($id): Response
    {
        return $this->response->addData(AttributeResource::make(Attribute::with([])->addRelations()->findOrFail($id)));
    }

    /**
     * Create a new Attribute.
     */
    public function store(StoreAttribute $request): JsonResponse
    {
        $inputs = $request->validated();
        // create a new attribute
        $attribute = new Attribute($inputs);
        $attribute->display = $request->input('display_options');
        $attribute->save();

        // create attribute values if has options
        //    if ( $attribute->has_options )
        //    {
        $optionValues = $inputs['option_values'] ?? [];
        foreach ($optionValues as $optionValue) {
            if (! empty($optionValue['value'] ?? null)) {
                $attribute->values()->create($optionValue);
            }
        }
        // load attribute values to return in response
        $attribute->load('values');

        return $this->response->success(Response::HTTP_CREATED, __('messages.success.create', ['resource' => $this->resource]))
            ->addData(AttributeResource::make($attribute));
    }

    /**
     * Update the attribute.
     *
     *
     * @throws Exception
     */
    public function update(StoreAttribute $request, $id): JsonResponse
    {
        $inputs = $request->validated();
        // save attribute
        $attribute = Attribute::with([])->findOrFail($id);
        $attribute->fill($inputs);
        $attribute->display = $request->has('display_options') ? $request->input('display_options') : $attribute->display;
        $attribute->save();

        // sync attribute values if has options
        if (isset($inputs['option_values'])) {
            // update or create attribute values
            $values = $inputs['option_values'] ?? [];
            foreach ($values as $option) {
                if (! empty($option['value'])) {
                    $attribute->values()->updateOrCreate(['value' => $option['value']], $option);
                }
            }
            // delete values that not return
            $attribute->values()->whereNotIn('value', array_column($values, 'value'))->delete();
            // load attribute values to return in response
        }
        $attribute->load('values');

        return $this->response
            ->setMessage(__('messages.success.update', [
                'resource' => $this->resource,
                'id' => $attribute->name,
            ]))
            ->addData(AttributeResource::make($attribute));
    }

    /**
     * Archive the attribute.
     */
    public function archive($attributeId): JsonResponse
    {
        $attribute = Attribute::with([])->findOrFail($attributeId);

        if ($attribute->archive()) {
            return $this->response
                ->setMessage(__('messages.success.archive', [
                    'resource' => $this->resource,
                    'id' => $attribute->name,
                ]))
                ->addData(AttributeResource::make($attribute));
        }

        return $this->response->warning()
            ->addWarning(__('messages.failed.already_archive', [
                'resource' => $this->resource,
                'id' => $attribute->name,
            ]), ucfirst($this->resource).Response::CODE_ALREADY_ARCHIVED, 'id', ['id' => $attributeId])
            ->addData(AttributeResource::make($attribute));
    }

    /**
     * Unarchived the attribute.
     */
    public function unarchived($id): JsonResponse
    {
        $attribute = Attribute::with([])->findOrFail($id);

        if ($attribute->unarchived()) {
            return $this->response
                ->setMessage(__('messages.success.unarchived', [
                    'resource' => $this->resource,
                    'id' => $attribute->name,
                ]))
                ->addData(AttributeResource::make($attribute));
        }

        return $this->response->warning()
            ->addWarning(__('messages.failed.unarchived', [
                'resource' => $this->resource,
                'id' => $attribute->name,
            ]), ucfirst($this->resource).Response::CODE_ALREADY_UNARCHIVED, 'id', ['id' => $id])
            ->addData(AttributeResource::make($attribute));
    }

    /**
     * Delete an attribute with its values and product attributes.
     *
     *
     * @throws Exception
     */
    public function destroy(Request $request, $id): JsonResponse
    {
        $attribute = Attribute::with([])->findOrFail($id);

        // check the attribute is used
        if (! boolval($request->get('ignore_relations', false)) && $related = $attribute->isUsed(true)) {
            return $this->response->error(Response::HTTP_BAD_REQUEST)
                ->setMessage(__('messages.failed.related_relations', [
                    'resource' => $this->resource,
                    'id' => $attribute->name,
                ]));
        }

        // delete the attribute with its relations
        $attribute->delete();

        return $this->response->setMessage(__('messages.success.delete', [
            'resource' => $this->resource,
            'id' => $attribute->name,
        ]));
    }

    /**
     * Show all attribute based on attribute groups.
     */
    public function grouped(): Response
    {
        $attributeGroups = AttributeGroup::with(['attributes'])->get();
        $unGroupedAttributes = Attribute::with([])
            ->whereNull('attribute_group_id')
            ->orWhere('attribute_group_id', 0)
            ->get();

        return $this->response->addData(AttributeGroupResource::collection($attributeGroups), 'grouped')
            ->addData($unGroupedAttributes, 'ungrouped');
    }

    /**
     * bulk archive using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_ARCHIVE);
    }

    /**
     * bulk un archive using request filters or body ids array.
     *
     *
     * @throws Exception
     */
    public function bulkUnArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_UN_ARCHIVE);
    }

    /**
     * Bulk delete attributes.
     *
     *
     * @throws Exception
     */
    public function bulkDestroy(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_DELETE);
    }

    /**
     * check the possibility of deletion.
     */
    public function isDeletable(Request $request): Response
    {
        // validate
        $request->validate([
            'ids' => 'required|array|min:1',
            'ids.*' => 'integer|exists:attributes,id',
        ]);

        $ids = array_unique($request->input('ids', []));

        $result = [];
        $attributes = Attribute::with([])->whereIn('id', $ids)->select('id', 'name')->get();
        foreach ($attributes as $key => $attribute) {
            $isUsed = $attribute->isUsed();

            $result[$key] = $attribute->only('id', 'name');
            $result[$key]['deletable'] = ! boolval($isUsed);
            $result[$key]['reason'] = $isUsed ?: null;
        }

        return $this->response->addData($result);
    }

    /**
     * {@inheritDoc}
     */
    protected function getModel()
    {
        return Attribute::class;
    }

    /**
     * {@inheritDoc}
     */
    protected function getResource()
    {
        return AttributeResource::class;
    }
}
