<?php

namespace App\Http\Controllers;

use App\DataTable\DataTable;
use App\Http\Resources\AccountingPaymentResource;
use App\Models\IntegrationInstance;
use App\Models\Payment;
use App\Repositories\IntegrationInstanceRepository;
use App\Response;
use Exception;
use Illuminate\Http\Request;
use Modules\Qbo\Manager\QboManager;
use Modules\Xero\Managers\XeroManager;

class AccountingPaymentController extends Controller
{
    use DataTable;

    private QboManager $qboManager;

    private XeroManager $manager;

    public function __construct(
        private readonly IntegrationInstanceRepository $integrationInstances
    ) {
        parent::__construct();

        if ($accountingInstance = $this->integrationInstances->getAccountingInstance()) {
            if ($accountingInstance->isQbo()) {
                $this->qboManager = new QboManager($accountingInstance);
            } elseif ($accountingInstance->isXero()) {
                $this->manager = app(XeroManager::class);
            }
        }
    }

    /**
     * @throws Exception
     */
    public function syncExternal(Request $request): Response
    {
        $request->validate([
            'ids' => 'required|array|min:1',
        ]);

        /** @var IntegrationInstance $accountingInstance */
        $accountingInstance = app(IntegrationInstanceRepository::class)->getAccountingInstance();

        if (! $accountingInstance) {
            return $this->response->addError('No accounting integration found', 404, 'message')->error(404);
        }

        $ids = $request->input('ids');

        if ($accountingInstance->isXero()) {
            $this->manager->syncPayments($ids);
        } elseif ($accountingInstance->isQbo()) {
            $this->qboManager->syncPayments($ids);
        }

        return $this->response->success(200);
    }

    /**
     * @throws Exception
     */
    public function deleteIntegrationPayments(Request $request): Response
    {
        $inputs = $request->validate([
            'ids' => 'required|array|min:1|exists:payments,id',
        ]);

        $this->manager->deletePayments($inputs['ids']);

        return $this->response->success(200);
    }

    /**
     * {@inheritDoc}
     */
    protected function getModel(): string
    {
        return Payment::class;
    }

    /**
     * {@inheritDoc}
     */
    protected function getResource(): string
    {
        return AccountingPaymentResource::class;
    }
}
