<?php

namespace App;

use App\DTO\DataTableFilterDto;
use App\DTO\DataTableFiltersDto;
use App\Lib\Inspect\Inspect;
use App\Models\Attribute;
use App\Models\FifoLayer;
use App\Models\Product;
use App\Models\Setting;
use App\Notifications\MonitoringMessage;
use Carbon\Carbon;
use Closure;
use Doctrine\DBAL\Schema\Index;
use Exception;
use Generator;
use Illuminate\Bus\BatchRepository;
use Illuminate\Contracts\Filesystem\FileNotFoundException;
use Illuminate\Database\QueryException;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Intervention\Image\Facades\Image;
use Laravel\Horizon\Contracts\JobRepository;
use SplFileObject;

//use Intervention\Image\Image;

class Helpers
{
    const OPERATION_UPDATE_CREATE = 'updateOrCreate';

    const OPERATION_DELETE = 'delete';

    const OPERATION_SET = 'set';

    const OPERATION_APPEND = 'append';

    public static $dieCount = 0;

    public static function dto2csvFile(Collection $records, ?callable $transformer_function = null): mixed
    {
        //Create temp file
        $delimiter = ',';
        $enclosure = '"';
        $escape_char = '\\';

        $tempFile = tmpfile();

        if (! $tempFile) {
            return false;
        }

        foreach ($records as $record) {
            fputcsv(
                $tempFile,
                is_null($transformer_function) ? $record : $transformer_function($record),
                $delimiter,
                $enclosure,
                $escape_char);
        }

        rewind($tempFile);

        return $tempFile;
    }

    public static function array2csvFile(array $records, ?callable $transformer_function = null): mixed
    {
        //Create temp file
        $delimiter = ',';
        $enclosure = '"';
        $escape_char = '\\';

        $tempFile = tmpfile();

        if (! $tempFile) {
            return false;
        }

        foreach ($records as $record) {
            fputcsv(
                $tempFile,
                is_null($transformer_function) ? $record : $transformer_function($record),
                $delimiter,
                $enclosure,
                $escape_char);
        }

        rewind($tempFile);

        return $tempFile;
    }

    /**
     * Covert csv string to associated array.
     */
    public static function csvToArray(string $csv_string, string $delimiter = ',', string $enclosure = '"'): array
    {
        $output_array = [];

        $csvLines = explode("\n", $csv_string);

        // Header from first line.
        $header = str_getcsv(trim($csvLines[0]), $delimiter, $enclosure);
        // loop on lines except first line.
        foreach (array_slice($csvLines, 1) as $line) {
            $fields_values = str_getcsv(mb_convert_encoding(trim($line), 'UTF-8', 'UTF-8'), $delimiter, $enclosure);
            $fields_values = self::sanitizeCsvRow($fields_values);
            if (count($fields_values) == count($header)) {
                // Combine header with values.
                $row = array_combine($header, $fields_values);
                // Add to output
                $output_array[] = $row;
            }
        }

        return $output_array;
    }

    /**
     * Read csv file as array.
     */
    public static function csvFileToArray(string|SplFileObject $file, string $delimiter = ',', string $enclosure = '"'): Generator
    {
        if (is_string($file)) {
            $file = new SplFileObject($file);
        }

        $file->setFlags(SplFileObject::READ_AHEAD | SplFileObject::SKIP_EMPTY | SplFileObject::DROP_NEW_LINE);
        $header = [];

        while (! $file->eof()) {
            $line = $file->fgetcsv($delimiter, $enclosure);

            if ($line) {
                if (empty($header)) {
                    $header = self::sanitizeCsvRow(self::encodeCsvRow($line));
                } else {
                    if (count($line) === count($header)) {
                        yield array_combine($header, self::sanitizeCsvRow(self::encodeCsvRow($line)));
                    }
                }
            }
        }
    }

    public static function csvFileToCollection(string|SplFileObject $file, string $delimiter = ',', string $enclosure = '"'): Collection
    {
        if (is_string($file)) {
            $file = new SplFileObject($file);
        }

        $file->setFlags(SplFileObject::READ_AHEAD | SplFileObject::SKIP_EMPTY | SplFileObject::DROP_NEW_LINE);
        $header = [];
        $rows = [];

        while (!$file->eof()) {
            $line = $file->fgetcsv($delimiter, $enclosure);

            if ($line) {
                if (empty($header)) {
                    $header = self::sanitizeCsvRow(self::encodeCsvRow($line));
                } else {
                    if (count($line) === count($header)) {
                        $rows[] = array_combine($header, self::sanitizeCsvRow(self::encodeCsvRow($line)));
                    }
                }
            }
        }

        return collect($rows);
    }

    public static function encodeCsvRow(array $line): array
    {
        return array_map(function ($current) {
            return mb_convert_encoding($current, 'UTF-8', 'UTF-8');
        }, $line);
    }

    /**
     * Gets the bytes value of size strings, e.g 2G.
     */
    public static function returnBytes($string): int
    {
        preg_match('/(?<value>\d+)(?<option>.?)/i', trim($string), $matches);
        $inc = [
            'g' => 1073741824, // (1024 * 1024 * 1024)
            'm' => 1048576, // (1024 * 1024)
            'k' => 1024,
        ];

        $value = (int) $matches['value'];
        $key = strtolower(trim($matches['option']));
        if (isset($inc[$key])) {
            $value *= $inc[$key];
        }

        return $value;
    }

    public static function sanitizeCsvRow(array $row): array
    {
        return array_map(function ($column) {
            return preg_replace('/[\x00-\x1F\x80-\xFF]/', '', $column);
        }, $row);
    }

    /**
     * Sync product attributes.
     */
    public static function getSyncingProductAttributes(array $attributes): array
    {
        /**
         * Get Existed attributes by attribute name.
         */
        $existed_attributes = Attribute::with([])->whereIn('name', array_column($attributes, 'name'))->get();

        /**
         * Fill syncing attributes.
         */
        $syncAttributes = [];
        foreach ($attributes as $product_attribute) {
            // Check if attribute exists
            $attribute = $existed_attributes->firstWhere('name', $product_attribute['name']);
            if (! $attribute) {
                // Create new attribute
                $attribute = new Attribute($product_attribute);
                $attribute->save();
            }

            // Add value to attribute values has options.
            if ($attribute->has_options) {
                $attribute->values()->updateOrCreate(['value' => $product_attribute['value']]);
            }
            // add to syncing attributes.
            $syncAttributes[$attribute->id] = ['value' => $product_attribute['value']];
        }

        return $syncAttributes;
    }

    /**
     * Calculate product average cost when new fifoLayer created.
     *
     * Basically any time a purchase is made, it takes the current avg cost * existing qty + (new qty * new cost)
     *  to total the value of the inventory then divide by total # of units
     */
    public static function updateProductAverageCostFifoLayer(int $productId, FifoLayer $newFifoLayer)
    {
        self::updateProductAverageCost($productId, $newFifoLayer->total_cost, $newFifoLayer->original_quantity);
    }

    /**
     * Calculate product average cost when custom fees created.
     */
    public static function updateProductAverageCostCustomFees(int $productId, float $customFees)
    {
        self::updateProductAverageCost($productId, $customFees);
    }

    /**
     * Update Product Average Cost.
     */
    public static function updateProductAverageCost(int $productId, $additionalCost, int $additionalQuantity = 0)
    {
        // Get product with active fifo layers
        $product = Product::with(['activeFifoLayers'])->findOrFail($productId);

        // Calculate average cost and update it.
        if ($product->getCumulativeAvailableInventory() + $additionalQuantity == 0) {
            $productAverageCost = 0;
        } else {
            $productAverageCost = ($product->getCumulativeCost() + $additionalCost) /
                            ($product->getCumulativeAvailableInventory() + $additionalQuantity);
        }

        $product->average_cost = $productAverageCost;
        $product->save();
    }

    /**
     * Retrieve image from url
     * This function check url is an image by content-type return <FALSE> if not image and
     * return array contains data and image extension.
     *
     *
     * @return array|bool <false> if content-type not image
     */
    public static function getImageFromUrl(string $imageUrl)
    {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($ch, CURLOPT_URL, $imageUrl);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 20);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        //    curl_setopt( $ch, CURLOPT_NOBODY, true );

        $content = curl_exec($ch);
        $contentType = curl_getinfo($ch, CURLINFO_CONTENT_TYPE); // content-type (mim-type) from response header
        curl_close($ch);

        // check content type is image
        $contentTypePairs = explode('/', $contentType);
        if ($contentTypePairs[0] != 'image') {
            return false;
        }

        // check content is image
        try {
            if (empty($content) || ! getimagesizefromstring($content)) {
                return false;
            }
        } catch (Exception $exception) {
            return false;
        }

        return ['data' => $content, 'extension' => $contentTypePairs[1]];
    }

    /**
     * Store image on "image" disk on the user directory.
     *
     * return URL of stored image
     */
    public static function storeImage($data, string $extension = 'jpg'): string
    {
        $fileName = uniqid();
        $fileDir = auth()->id().date('/Y/m/');

        $filePath = $fileDir.$fileName.'.'.$extension;
        $smallFilePath = $fileDir.config('image.small_image_prefix').$fileName.'.'.$extension;

        // store main image
        $imageStorage = Storage::disk('images');
        $imageStorage->put($filePath, $data);

        // resize small image
        $smallImage = Image::make($imageStorage->path($filePath));
        $smallImage->fit(config('image.small_image_width'));
        $smallImage->encode();

        // store small image
        $imageStorage->put($smallFilePath, $smallImage->getEncoded());

        // return main image path
        return $filePath;
    }

    /**
     * Delete image with its thumbnail.
     */
    public static function deleteImage(string $path): bool
    {
        // remove base storage path from image path
        $baseImageStorageUrl = Storage::disk('images')->url('');
        if (Str::startsWith($path, $baseImageStorageUrl)) {
            $path = ltrim(Str::replaceFirst($baseImageStorageUrl, '', $path), '/');
        }

        // get small image path
        $smallImagePath = explode('/', $path);
        $fileNameIndex = count($smallImagePath) - 1;
        $smallImagePath[$fileNameIndex] = config('image.small_image_prefix').$smallImagePath[$fileNameIndex];
        $smallImagePath = implode('/', $smallImagePath);

        // delete images
        return Storage::disk('images')->delete([$path, $smallImagePath]);
    }

    /**
     * Get additional array to retrieve with the collection response.
     *
     * @param $model
     *
     * @return array
     */
    public static function getResponseAdditional($model)
    {
        $additional = [];

        $model = new $model();

        // instance of Model to get casting columns
        if ($model instanceof Filterable and $model instanceof Model) {
            $additional = [
                'available_columns'  => $model->availableColumns(),
                'casting_columns'    => $model->getCasts(),
                'filterable_columns' => $model->filterableColumns(),
            ];
        }

        // Sortable
        if ($model instanceof Sortable) {
            $additional['sortable_columns'] = $model->sortableColumns();
        }

        return $additional;
    }

    /**
     * Get Setting by key.
     *
     * @param  null  $default
     * @return mixed|null
     */
    public static function setting($key, $default = null, $useCache = false)
    {
        try {
            if (isset(Setting::$fetchedSettings[$key]) && $useCache) {
                return Setting::$fetchedSettings[$key];
            }

            // get all settings with this key
            $settings = Setting::with([])->where('key', $key)->get();

            // if exists
            if ($settings->isNotEmpty()) {
                // get setting associated with this user or general if the user not have owned setting
                $setting = $settings->firstWhere('user_id', auth()->user()) ?: $settings->firstWhere('user_id', null);

                if ($setting) {
                    // return default_value if actual value is null
                    // $value = $setting->value ?: $setting->default_value;
                    $value = $setting->value; //SKU-3716

                    // add to fetched settings
                    Setting::$fetchedSettings[$key] = $value;

                    return $value;
                }
            }
        } catch (\Exception $e) {
            // Too sensitive of a function to let it throw exceptions.
            // Just return default value if it fails.
        }

        // return default if setting not exists
        return $default;
    }

    /**
     * Retrieve image url.
     *
     * Check if base64, store image and return stored image url else, store or return url.
     *
     *
     * @return string|UploadedFile|null <null> if invalid image url
     *
     * @throws FileNotFoundException
     */
    public static function getImageUrl(string $image, bool $download = false)
    {
        if (! $image) {
            return null;
        }

        // upload by file
        if ($image instanceof UploadedFile) {
            return self::storeImage($image->get(), $image->extension());
        }

        // add image (upload)
        if (preg_match('/^data:image\/(\w+);base64,/', $image, $match)) {
            $data = substr($image, strpos($image, ',') + 1);
            $data = base64_decode($data);

            $extension = $match[1] ?? 'jpg';

            return self::storeImage($data, $extension);
        } else {
            $isExternalUrl = isset(parse_url($image)['scheme']); //filter_var( trim($image), FILTER_VALIDATE_URL, FILTER_FLAG_PATH_REQUIRED );
            if ($isExternalUrl) {
                // resolve image
                $imageContent = self::getImageFromUrl($image);

                // return <FALSE> invalid image url
                if (! $imageContent) {
                    return false;
                }

                // download image from url
                if ($download) {
                    return self::storeImage($imageContent['data'], $imageContent['extension']);
                }

                // return image without resolving
                return $image;
            }

            // add image from local storage
            // remove base storage path from image url
            $baseImageStorageUrl = Storage::disk('images')->url('');
            if (Str::startsWith($image, $baseImageStorageUrl)) {
                $image = ltrim(str_replace($baseImageStorageUrl, '', $image), '/');
            }

            // check if image exists
            if (! Storage::disk('images')->exists($image)) {
                return false;
            }

            return $image;
        }
    }

    /**
     * Get number in the correct format.
     */
    public static function getDecimalAttribute($number, int $leadingZero = 2, int $digits = 4): string
    {
        //    default without leading zero
        //    $total = $leadingZero + $digits + 1;
        //    return sprintf( "%0{$total}.{$digits}f", $number );

        return sprintf("%.{$digits}f", $number);
    }

    /**
     * Get phone number with country prefix.
     *
     * @param \App\Models\Constant\ConstantsCountry null $country
     */
    public static function getPhoneAttribute(string $phoneNumber, ?Models\Constant\ConstantsCountry $country = null): string
    {
        // TODO: Fix N+1 query here
        /*$country = $country ?: \App\Models\Constant\ConstantsCountry::with([])->where('code', self::setting(Setting::KEY_SELLER_COUNTRY, 'US'))->first();

        if (is_null($country)) {
            return null;
        }

        $numberLength = strlen($phoneNumber);

        if ($numberLength == 9 || $numberLength == 10) {
            $phoneNumber = $country->phone_code.$phoneNumber;
        }*/

        return $phoneNumber;
    }

    /**
     * Add prefix to array keys.
     */
    public static function addPrefixToArrayKeys(array $array, string $prefix): array
    {
        $result = [];
        foreach ($array as $key => $value) {
            $result["{$prefix}.{$key}"] = $value;
        }

        return $result;
    }

    /**
     * Save description of passed object's API (as an HTML file in /doc folder).
     */
    public static function saveDoc(object $object, string $file_name, $params = [])
    {
        $append = ! empty($params['append']);
        static $saved_doc_count = 0;
        // Helper function for developers:
        // Helpers::saveDoc( $object, "NameOfClass" );
        // Will save /doc/NameOfClass.html describing the API of the passed object.

        file_put_contents(base_path().'/doc/'.$file_name.'.html', Inspect::getHtml($object), ($append ? FILE_APPEND : 0));
        @header('X-Saved-Doc-'.$saved_doc_count.': '.\App::make('url')->to('/').'/doc/'.$file_name.'.html');
        $saved_doc_count++;
        echo '<div>API info saved in <a href="/doc/'.$file_name.'.html">/doc/'.$file_name.'.html</a></div>';
    }

    /**
     * Save description of passed object's API (as an HTML file in /doc folder).
     */
    public static function addToDoc(object $object, string $file_name)
    {
        return static::saveDoc($object, $file_name, ['append' => 1]);
    }

    public static function checkpoint($name = '')
    {
        static $counter = 0;
        $counter++;
        $backtrace = debug_backtrace(0, 1);
        //dd($backtrace);
        $file = $backtrace[0]['file'];
        $line = $backtrace[0]['line'];
        header('X-checkpoint-'.$counter.': '.$name.' ('.$file.':'.$line.')');
    }

    public static function isJobRunning($job): bool
    {
        $jobRepository = app()->make(JobRepository::class);

        return (bool) $jobRepository->getRecent()->where('name', $job)->whereIn('status', [
            'reserved',
            'pending',
        ])->count();
    }

    public static function isBatchRunning($name): bool
    {
        $batches = app()->make(BatchRepository::class)->get();

        foreach ($batches as $batch) {
            if (! $batch->finished() && ! $batch->cancelled() && $batch->name == $name) {
                //$batch->cancel();
                //Log::info('An unfinished batch for '.$name.' is still pending, so cancelling');
                return true;
            }
        }

        return false;
    }

    public static function printToStderr($msg)
    {
        fwrite(STDERR, $msg);
    }

    public static function getAppTimezone(): ?string
    {
        $setting = Setting::with([])->where('key', Setting::KEY_DEFAULT_TIMEZONE)
            ->first();

        return $setting?->value;
    }

    public static function setAppTimezone()
    {
        $timezone = self::getAppTimezone();
        if ($timezone) {
            date_default_timezone_set($timezone);
        }
    }

    public static function timeToTimezone(string $time, string $fromTimezone, string $toTimezone): string
    {
        return now($fromTimezone)->setTimeFromTimeString($time)->timezone($toTimezone)->format('H:i');
    }

    public static function dateLocalToUtc(Carbon|string $date): Carbon
    {
        return Carbon::parse($date, self::setting(Setting::KEY_DEFAULT_TIMEZONE))->setTimezone('UTC');
    }

    public static function dateUtcToLocal(Carbon|string $date): Carbon
    {
        return Carbon::parse($date, 'UTC')->setTimezone(self::setting(Setting::KEY_DEFAULT_TIMEZONE));
    }

    public static function utcStartOfLocalDate(Carbon|string $date): Carbon
    {
        return self::dateLocalToUtc(self::dateUtcToLocal($date)->startOfDay());
    }

    public static function utcEndOfLocalDate(Carbon|string $date): Carbon
    {
        return self::dateLocalToUtc(self::dateUtcToLocal($date)->endOfDay());
    }

    public static function utcToStartOfLocalDateInUtc(Carbon|string $date): Carbon
    {
        return Carbon::parse($date)->setTimezone(self::getAppTimezone())->startOfDay()->setTimezone('UTC');
    }

    public static function getSqlUtcConvertToStartOfLocalDate(string $fieldName): string
    {
        return 'DATE_FORMAT(
                    CONVERT_TZ(
                        '.$fieldName.', "UTC", "'.self::setting(Setting::KEY_DEFAULT_TIMEZONE).'"
                    ), "%Y-%m-%d 00:00:00"
                )';
    }

    public static function getSqlUtcStartOfLocalDate(string $fieldName): string
    {
        return 'CONVERT_TZ('.self::getSqlUtcConvertToStartOfLocalDate($fieldName).
            ', "'.self::setting(Setting::KEY_DEFAULT_TIMEZONE).'", "UTC")';
    }

    /**
     * TODO: The way the condition is build is not ideal.  It only passes if the current time is within 2 minutes of the scheduled time
     */
    public static function isScheduleDue(array $schedule): bool
    {
        if (empty($schedule)) {
            return false;
        }

        $now = now(static::getAppTimezone());
        $todayString = strtolower($now->format('l'));
        $matchingSchedule = array_values(array_filter($schedule, function ($row) use ($now, $todayString) {
            $timeByUserTimezone = static::timeToTimezone($row['time'], config('app.timezone'), static::getAppTimezone());
            $scheduleTime = $now->clone()->setTimeFromTimeString($timeByUserTimezone);

            return in_array($todayString, $row['days'] ?? []) &&
                   $now->isBetween($scheduleTime, $scheduleTime->clone()->addMinutes(2));
        }));

        return ! empty($matchingSchedule);
    }

    public static function weightConverter($value, $fromUnit, $toUnit)
    {
        if (empty($value)) {
            return null;
        }

        if (! $fromUnit || ! $toUnit) {
            return null;
        }

        if ($fromUnit == $toUnit) {
            return $value;
        }

        $conversions = [
            Product::WEIGHT_UNIT_KG => [
                Product::WEIGHT_UNIT_LB => 2.204623,
                Product::WEIGHT_UNIT_OZ => 35.27396,
                Product::WEIGHT_UNIT_G => 1000,
            ],
            Product::WEIGHT_UNIT_LB => [
                Product::WEIGHT_UNIT_KG => 0.4535924,
                Product::WEIGHT_UNIT_OZ => 16,
                Product::WEIGHT_UNIT_G => 453.592,
            ],
            Product::WEIGHT_UNIT_OZ => [
                Product::WEIGHT_UNIT_KG => 0.02834952,
                Product::WEIGHT_UNIT_LB => 0.0625,
                Product::WEIGHT_UNIT_G => 28.3495,
            ],
            Product::WEIGHT_UNIT_G => [
                Product::WEIGHT_UNIT_KG => 0.001,
                Product::WEIGHT_UNIT_LB => 0.00220462,
                Product::WEIGHT_UNIT_OZ => 0.035274,
            ],
        ];

        return $value * $conversions[$fromUnit][$toUnit];
    }

    public static function dimensionConverter($value, $fromUnit, $toUnit)
    {
        if (empty($value)) {
            return null;
        }

        if ($fromUnit == $toUnit) {
            return $value;
        }

        $dimensionUnitMeter = 'm';

        $conversions = [
            Product::DIMENSION_UNIT_MM => [
                Product::DIMENSION_UNIT_INCH => 0.0393701,
                $dimensionUnitMeter => 0.001,
            ],
            Product::DIMENSION_UNIT_CM => [
                Product::DIMENSION_UNIT_INCH => 0.3937008,
                $dimensionUnitMeter => 0.01,
            ],
            Product::DIMENSION_UNIT_INCH => [
                Product::DIMENSION_UNIT_CM => 2.54,
                $dimensionUnitMeter => 0.0254,
            ],
        ];

        return $value * $conversions[$fromUnit][$toUnit];
    }

    public static function dieAt($times, $var = '')
    {
        echo $var;
        static::$dieCount++;
        if (static::$dieCount >= $times) {
            exit();
        }
    }

    // Very basic lazy iterator functionality:
    // (for advanced functionality we could use https://github.com/nikic/iter)

    // replace array_map($arr, $cb) with \App\Helpers::iteratorMap($iter, $cb)
    public static function iteratorMap(callable $callback)
    {
        $iterators = array_slice(func_get_args(), 1);

        $multi = new \MultipleIterator;
        foreach ($iterators as $it) {
            $multi->attachIterator($it);
        }

        foreach ($multi as $current) {
            yield call_user_func_array($callback, $current);
        }
    }

    public static function iteratorFilter(callable $callback)
    {
        $iterators = array_slice(func_get_args(), 1);

        $multi = new \MultipleIterator;
        foreach ($iterators as $it) {
            $multi->attachIterator($it);
        }

        foreach ($multi as $current) {
            if (call_user_func_array($callback, $current)) {
                yield $current;
            }
        }
    }

    // End of basic lazy iterator functionality

    /**
     * Get file as base64 string.
     *
     *
     * @return string|null|false
     */
    public static function getBase64OfImage(?string $file, ?int $resizeWidth = null)
    {
        if (empty($file)) {
            return null;
        }

        try {
            $isExternal = isset(parse_url($file)['scheme']);
            if (! $isExternal) {
                $file = public_path($file);
            }

            $image = Image::make($file);
            if ($resizeWidth) {
                $image->widen($resizeWidth, function ($constraint) {
                    $constraint->upsize();
                });
            }

            return base64_encode($image->encode('png')->getEncoded());
        } catch (\Throwable $exception) {
            return false;
        }
    }

    /**
     * Trying to handle "Duplicate entry" and "Deadlock" SQL exceptions
     *
     * Do NOT use this function unless you are sure that the closure checks the duplication before inserting a new row
     */
    public static function handleDuplicateAndDeadlockSQLExceptions(Closure $closure): mixed
    {
        // try 5 times
        foreach (range(1, 5) as $attempt) {
            try {
                return $closure();
            } catch (QueryException $queryException) {
                // SQLSTATE[23000]: Identify constraint violation: 1062 Duplicate entry
                // SQLSTATE[40001]: Serialization failure: 1213 Deadlock found when trying to get lock
                $sqlState = $queryException->errorInfo[0] ?? 0;
                if (($sqlState != 23000 && $sqlState != 40001) || $attempt == 5) {
                    throw $queryException;
                }
                usleep(100); // 100 microseconds
            }
        }
    }

    public static function arrayDiffAssoc(array $array, array $other): array
    {
        $array = Arr::sortRecursive($array);
        $other = Arr::sortRecursive($other);

        $diff = array_diff_assoc(array_map('serialize', $array), array_map('serialize', $other));

        return array_map('unserialize', $diff);
    }

    public static function arrayDiffAssoc2(array $array, array $other): array
    {
        $array = Arr::sortRecursive($array);
        $other = Arr::sortRecursive($other);

        $array = array_map('json_encode', $array);
        $other = array_map('json_encode', $other);

        $diff = array_diff_assoc($array, $other);

        return array_map('json_decode', $diff);
    }

    /**
     * Log mysql Deadlock exception with concurrent queries
     */
    public static function logMysqlDeadlockException(QueryException $exception): void
    {
        if (config('logging.log_mysql_deadlock_exceptions')) {
            $message = class_basename($exception->getFile()).':'.$exception->getMessage();
            $concurrentQueries = array_map(fn ($q) => (array) $q, DB::select('SHOW FULL PROCESSLIST'));

            $concurrentQueries = array_filter($concurrentQueries, function ($query) {
                return $query['Command'] != 'Sleep';
            });

            $concurrentQueries = array_map(function ($query) {
                $query['Info'] = str_replace("\n", '', $query['Info']);

                return $query;
            }, $concurrentQueries);

            $engineInnodbStatus = array_map(fn ($q) => (array) $q, DB::select('SHOW ENGINE INNODB STATUS'));

            Log::channel('deadlock_exception')->error($message, ['queries' => $concurrentQueries, 'status' => $engineInnodbStatus]);
        }
    }

    public static function schemaHasForeignKey(Blueprint $table, string $column)
    {
        $foreignKeys = Schema::getConnection()
            ->getDoctrineSchemaManager()
            ->listTableForeignKeys($table->getTable());

        return collect($foreignKeys)->map(function ($fk) {
            return $fk->getColumns();
        })->flatten()->contains($column);
    }

    public static function schemaHasIndex(string $table, string|array $columns, bool $unique = false): ?Index
    {
        $columns = Arr::wrap($columns);
        $indexes = Schema::getConnection()->getDoctrineSchemaManager()->listTableIndexes($table);

        return collect($indexes)
            ->filter(fn (Index $i) => ! $unique || $i->isUnique())
            ->filter(fn (Index $i) => count($i->getColumns()) == count($columns))
            ->filter(fn (Index $i) => empty(array_diff($i->getColumns(), $columns)))
            ->first();
    }

    public static function slack(string $message): void
    {
        Notification::route('slack', config('slack.alerts'))->notify(new MonitoringMessage($message));
    }

    public static function carbonizeDates($data, $date_fields): array
    {
        foreach ($date_fields as $date_field) {
            $data[$date_field] = isset($data[$date_field]) ? Carbon::parse($data[$date_field]) : null;
        }

        return $data;
    }

    public static function array2multidimensional(array $array, string $separator): array
    {
        $options = [];
        foreach ($array as $item) {
            $exploded_result = explode($separator, $item);
            $options[$exploded_result[0]] = $exploded_result[1];
        }

        return $options;
    }

    /**
     * Parse "Link" header which is returned by some APIs including the WooCommerces API
     * https://www.rfc-editor.org/rfc/rfc5988#section-5
     *
     * @param  string  $linkHeader example: <https://www.myapi.com/mydata&page=1>; rel="prev", <https://www.myapi.com/mydata&page=3>; rel="next"
     */
    private static function parseLinkHeader(string $linkHeader): array
    {
        /*
         * TODO: Return just the page number if "next" exists, null if not
         */

        preg_match_all('~<(.*)>.*rel="(.*)"~iU', $linkHeader, $matches);

        return array_combine($matches[2], $matches[1]);
    }

    public static function getPageNumberFromLinkHeader(string $linkHeader): ?int
    {
        $links = self::parseLinkHeader($linkHeader);

        if (isset($links['next'])) {
            $url = $links['next'];
            $urlParts = parse_url($url);
            parse_str($urlParts['query'], $query);

            return $query['page'] ?? null;
        }

        return null;
    }

    public static function unixTimestampToCarbon($dateString): Carbon
    {
        $timestamp = preg_replace('/[^0-9]/', '', strstr($dateString, '+', true));

        // Convert the timestamp to seconds and create a Carbon instance
        return Carbon::createFromTimestamp($timestamp / 1000);
    }

    public static function simpleFilter($field, $value): string
    {
        return 'filters='.
            json_encode(
                DataTableFiltersDto::from([
                    'filterSet' => [
                        DataTableFilterDto::from([
                            'column' => $field,
                            'operator' => '=',
                            'value' => $value,
                        ]),
                    ],
                ])->toArray(),
            );
    }
}
