<?php

namespace App;

use App\Exceptions\InventoryMovementTypeException;
use App\Exceptions\SupplierWarehouseCantHaveInventoryMovementsException;
use App\Models\InventoryMovement;
use App\Models\SalesOrderLine;
use App\Repositories\InventoryMovementRepository;
use App\Services\StockTake\OpenStockTakeException;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Database\MultipleRecordsFoundException;

class FixOrphanSalesOrderLineReservationMovementUsingReference
{
    protected InventoryMovement $inventoryMovement;
    protected SalesOrderLine $salesOrderLine;

    public function __construct(
        private readonly InventoryMovementRepository $movements,
    )
    {
    }

    /**
     * @throws InventoryMovementTypeException
     * @throws OpenStockTakeException
     * @throws SupplierWarehouseCantHaveInventoryMovementsException
     */
    public function __invoke(
        InventoryMovement $inventoryMovement
    ): ?SalesOrderLine
    {
        $this->inventoryMovement = $inventoryMovement;
        if (!$this->findSalesOrderLine()) {
            return null;
        }
        $this->updateLink();

        return $this->salesOrderLine->refresh();
    }

    private function findSalesOrderLine(): bool
    {
        try {
            $this->salesOrderLine = $this->movements->findSalesOrderLineMatchUsingReference($this->inventoryMovement);
        } catch (ModelNotFoundException) {
            return false;
        } catch (MultipleRecordsFoundException) {
            customlog('inventory-fixes', "Multiple sales order lines found for movement ID {$this->inventoryMovement->id} reference {$this->inventoryMovement->reference}", days: null);
            return false;
        }
        return true;
    }

    /**
     * @throws InventoryMovementTypeException
     * @throws OpenStockTakeException
     * @throws SupplierWarehouseCantHaveInventoryMovementsException
     */
    private function updateLink(): void
    {
        $this->inventoryMovement->link_id = $this->salesOrderLine->id;
        $this->inventoryMovement->save();
    }
}
