<?php
/**
 * Created by PhpStorm.
 * User: brightantwiboasiako
 * Date: 12/10/20
 * Time: 10:41 PM.
 */

namespace App\Exporters;

use App\Exporters\Formatters\ExportFormatterFactory;

/**
 * Class BaseExporter.
 */
class BaseExporter
{
    /**
     * The records to be exported.
     *
     * @var array
     */
    protected $records;

    /**
     * @var string
     */
    protected $format;

    /**
     * @var string
     */
    protected $filename;

    /**
     * @var array
     */
    protected $headers = [];

    /**
     * @var array
     */
    protected $options = [];

    /**
     * @var array Specifies a filtering & mapping of fields to be exported.
     */
    protected $exportableFieldsMapping = [];

    /**
     * BaseExporter constructor.
     */
    public function __construct(array $records, string $format, string $filename, array $headers = [], array $options = [])
    {
        $this->records = $records;
        $this->format = $format;
        $this->filename = $filename;
        $this->headers = ! empty($this->headers) ? $this->headers : $headers;
        $this->options = $options;
    }

    public function getFilename(): string
    {
        return $this->filename;
    }

    /**
     * Exports the records to the given format.
     */
    public function export(): string
    {
        /**
         * If a mapping exists for the fields to be exported,
         * we map and handle any re-write.
         */
        return ExportFormatterFactory::make($this->format)
            ->export(
                $this->filename,
                $this->headers,
                $this->records,
                $this->options
            );
    }

    public function setExportableFieldsMapping(array $mapping): self
    {
        $this->exportableFieldsMapping = $mapping;

        return $this;
    }

    /**
     * Adapts field mapping to the data to be exported.
     */
    public function adaptFieldMapping()
    {
        /**
         * The keys in the mapping are the field names
         * and the values are the re-writes.
         */
        $headers = array_values(array_intersect($this->headers, array_keys($this->exportableFieldsMapping)));
        // Get the records based on the mapped headers
        $records = array_map(function ($record) use ($headers) {
            $record = array_combine(array_keys($record), array_values($record));

            return array_intersect_key($record, array_flip($headers));
        }, $this->records);

        // Now we re-write the headers and keys
        $this->headers = [];

        foreach ($headers as $header) {
            $this->headers[] = $this->exportableFieldsMapping[$header];
        }

        $this->records = array_map(function ($record) {
            return array_combine($this->headers, $record);
        }, $records);
    }

    public static function groupByLines(array $data, string $groupingField, string $itemsKey = 'item_info'): array
    {
        // TODO: Make sure to always include Sales order number (for all rows) and only show id if selected.

        if (empty($data)) {
            return $data;
        }

        // If items are not being exported, we simply return
        // the data given
        $keys = array_keys($data[0]);

        if (! in_array($itemsKey, $keys)) {
            return $data;
        }

        // We create a row for each line.
        $transformed = [];

        foreach ($data as $order) {
            foreach ($order[$itemsKey] as $key => $orderLine) {
                unset($order[$itemsKey]);

                // The first line must have all records
                if ($key === 0) {
                    $orderDataForRow = $order;
                } else {
                    // Other records
                    $orderDataForRow = [];
                    foreach ($order as $k => $value) {
                        if ($k === $groupingField) {
                            $orderDataForRow[$k] = $value;
                        } else {
                            $orderDataForRow[$k] = null;
                        }
                    }
                }

                $transformed[] = array_merge($orderDataForRow, $orderLine);
            }
        }

        return $transformed;
    }
}
