<?php

namespace App\Events;

use Illuminate\Broadcasting\Channel;
use Illuminate\Broadcasting\InteractsWithSockets;
use Illuminate\Contracts\Broadcasting\ShouldBroadcastNow;
use Illuminate\Foundation\Events\Dispatchable;
use Illuminate\Queue\SerializesModels;
use Throwable;

/**
 * Class SKUProgressEvent
 */
class SKUProgressEvent implements ShouldBroadcastNow
{
    use Dispatchable, InteractsWithSockets, SerializesModels;

    const SKUProgressEventPOReceipt = 'po-receipt';

    protected string $channel;

    public int $total;

    public int $completed;

    public array $payload = [];

    public array $errors = [];

    /**
     * Create a new event instance.
     *
     * @return void
     */
    public function __construct(string $channel, int $total, int $completed = 0)
    {
        $this->channel = $channel;
        $this->total = $total;
        $this->completed = $completed;
    }

    public function broadcastWith(): array
    {
        return [
            'total' => $this->total,
            'completed' => $this->completed,
            'payload' => $this->payload,
            'errors' => $this->errors,
        ];
    }

    public function increment(int $by = 1): self
    {
        if ($this->completed + $by <= $this->total) {
            $this->completed += $by;
        }

        return $this;
    }

    public function advance(int $by = 1): self
    {
        return $this->increment($by)->send();
    }

    public static function sendOn(string $channel, int $total, int $completed = 0): self
    {
        return (new static($channel, $total, $completed))->send();
    }

    public function addError($key, string $message, $meta = null): self
    {
        $this->errors[] = [
            'key' => $key,
            'message' => $message,
            'meta' => $meta,
        ];

        return $this;
    }

    public function clearErrors(): self
    {
        $this->errors = [];

        return $this;
    }

    public function send(): self
    {
        try {
            broadcast($this);
        } catch (Throwable) {
        }

        return $this;
    }

    public function broadcastOn(): Channel|array
    {
        return new Channel($this->channel);
    }
}
