<?php

namespace App\DataTable;

use Exception;
use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\JsonResource;
use Illuminate\Support\Facades\DB;

class NewDataTableResource extends JsonResource
{
    protected bool $hasId = true;

    /**
     * Overrides the included fields.
     *
     * @var null
     */
    public static $includeOverride = null;

    /**
     * Excluded keys.
     *
     * @var array
     */
    protected $except;

    protected $include;

    protected $subExcept;

    protected $subInclude;

    protected $persistentFields = [];

    /**
     * Indicates if user has explicitly asked for excluded = [].
     * This is treated as including everything.
     *
     * @var bool
     */
    protected $hasEmptyExcluded = false;

    public function __construct($resource)
    {
        parent::__construct($resource);

        $this->except = json_decode(urldecode(request('excluded', '[]')), true);
        $this->include = json_decode(urldecode(request('included', '[]')), true);

        is_array($this->except) ?: $this->except = [];
        is_array($this->include) ?: $this->include = [];

        $this->hasEmptyExcluded = ! is_null(request('excluded')) && empty($this->except);

        if ($this->resource) {
            $this->persistentFields = array_keys(DataTableConfiguration::getPersistentRelations(get_class($this->resource)));
        }

        if (empty($this->include) && self::$includeOverride) {
            $this->include = self::$includeOverride;
        }

        $this->subExcept = $this->getSub($this->except);
        $this->subInclude = $this->getSub($this->include, true);
    }

    public function isIncluded(string $key): bool
    {
        return in_array($key, $this->include);
    }

    /**
     * Determined whether the key not in excluded keys.
     *
     * return the value if the key not in excluded keys
     * return MissingValue if in excludes keys
     */
    public function inclusive(string $key, $value): mixed
    {
        /**
         * Evaluate the conditions before executing the closure to avoid N+1 query problem.
         */
        $exceptCondition = empty($this->include) && ! in_array($key, $this->except);
        $includeCondition = empty($this->except) && (in_array($key, $this->include) || array_key_exists($key, $this->subInclude));
        $persistentCondition = in_array($key, $this->persistentFields);

        // Check if conditions are met before evaluating the closure
        if ($exceptCondition || $includeCondition || $persistentCondition || $this->hasEmptyExcluded) {
            // Now evaluate the closure if it's a closure
            $value = $value instanceof \Closure ? value($value) : $value;

            // Process sub-keys
            if (array_key_exists($key, $this->subExcept)) {
                try {
                    $value = collect($value)->except($this->subExcept[$key]);
                } catch (Exception $exception) {
                    // Handle exception
                }
            }

            if (array_key_exists($key, $this->subInclude)) {
                try {
                    $value = collect($value)->only($this->subInclude[$key]);
                } catch (Exception $exception) {
                    // Handle exception
                }
            }
        }

        return $this->when($exceptCondition || $includeCondition || $persistentCondition || $this->hasEmptyExcluded, $value);
    }

    /**
     * Get sub columns from array.
     */
    public function getSub(array $except, bool $forOnly = false): array
    {
        $subExcept = [];
        foreach ($except as $item) {
            if (count($subs = explode('.', $item)) > 1) {
                if (! isset($subExcept[$subs[0]])) {
                    $subExcept[$subs[0]] = [];
                }

                $subExcept[$subs[0]][] = $forOnly ? $subs[1] : implode('.', array_splice($subs, 1));
            }
        }

        return $subExcept;
    }

    public static function collectionWithTableSpecifications($resource, $model)
    {
        $additional = request('table_specifications') == 2 ? DataTableConfiguration::getTableSpecifications($model) : [];
        $additional['status'] = __('messages.status_success');

        // return queries for testing
        if (! app()->runningUnitTests() && ! app()->isProduction()) {
            $additional['queries'] = DB::getQueryLog();
        }

        return parent::collection($resource)->additional($additional);
    }

    /*
     * Use custom columns for two reasons:
     *
     * 1. Columns not available in data table that should be available in single entity view
     * 2. Columns that are more complex of a definition than $this->inclusive($column, $this->{$column})
     */
    public function getCustomColumns(): array
    {
        // Custom columns is optional, so by default we just return an empty array
        return [];
    }

    public final function toArray(Request $request): array
    {
        $columns = array_merge(
            $this->hasId && isset($this->id) ? ['id' => $this->id] : [],
            $this->getCustomColumns()
        );
        return array_merge(
            $columns,
            $this->generateColumnsFromDataTableConfig($columns)
        );
    }

    private function generateColumnsFromDataTableConfig(array &$columns): array
    {
        foreach ($this->resource->availableColumns() as $column => $type) {
            if (!isset($columns[$column])) {
                $columns[$column] = $this->inclusive($column, $this->{$column});
            }
        }

        return $columns;
    }
}
