<?php

namespace App\DataTable\Exports;

use App\Exceptions\ExportingEmptyDataException;
use App\Exporters\BaseExporter;
use App\Exporters\Formatters\ExportFormatterFactory;
use Illuminate\Support\Arr;
use Illuminate\Support\Str;

class DataTableExporter extends BaseExporter
{
    const MAX_SYNC_EXPORT_SIZE = 20000;

    /**
     * @var array The data to be exported
     */
    private array $data;

    /**
     * @var int|null|string The id of the user exporting the data
     */
    private int|null|string $userId;

    private ?array $includedFields = [];

    /**
     * DataTableExporter constructor.
     *
     * @throws ExportingEmptyDataException
     */
    public function __construct(
        array $data,
        string $resourceName,
        ?string $format,
        ?int $userId = null,
        ?array $mapping = [],
        ?array $included = []
    ) {
        if (empty($data)) {
            throw new ExportingEmptyDataException('Exporting empty data');
        }

        $this->includedFields = $included;
        $this->userId = $userId ?? auth()->id();
        // Set the fields mapping if available.
        $this->setExportableFieldsMapping($mapping ?? []);
        $this->data = $data;

        $this->headers = $this->reArrangeHeaders($this->makeHeaders());

        parent::__construct(
            $this->makeRecords(),
            $format,
            $this->makeFilename($resourceName, $format ?? ExportFormatterFactory::EXPORT_FORMAT_CSV)
        );
    }

    protected function makeHeaders(): array
    {
        return array_merge([
            'id' => 'id', // We always add the id field for imports, for instance.
        ],
            array_filter($this->exportableFieldsMapping, function ($field, $key) {
                foreach ($this->includedFields as $included) {
                    if (Str::startsWith($key, $included)) {
                        return true;
                    }
                }

                return false;
            }, ARRAY_FILTER_USE_BOTH));
    }

    public function reArrangeHeaders($headers)
    {
        $result = ['id' => 'id'];
        foreach ($this->includedFields as $included) {
            foreach ($headers as $key => $field) {
                if (Str::startsWith($key, $included)) {
                    $result[$key] = $field;
                }
            }
        }

        return $result;
    }

    public function append(string $fileName, array $records)
    {
        $handle = fopen($fileName, 'a');
        $this->data = $records;
        fputcsv($handle, $this->makeRecords());
        fclose($handle);
    }

    public function makeRecords(): array
    {
        $data = $this->data;

        return array_map(function ($row) {
            $parsedRow = [];
            foreach ($this->headers as $key => $header) {
                $parsedRow[$header] = $this->sanitize(Arr::get($row, $key));
            }

            return $parsedRow;
        }, $data);
    }

    /**
     * @return string|array|mixed
     */
    protected function sanitize($value)
    {
        if ($value === null) {
            $value = '';
        }
        if (is_array($value)) {
            $value = implode(' ', $value);
        }

        return $value;
    }

    protected function isDateFormat($value): bool
    {
        return (bool) preg_match('/^[0-9]{4}-(0[1-9]|1[0-2])-(0[1-9]|[1-2][0-9]|3[0-1])/', $value);
    }

    public function getRecords(): array
    {
        return $this->records;
    }

    protected function makeFilename(string $resourceName, string $format): string
    {
        return intval(date('YmdHis').rand(0, 99999)).'_'.$this->userId.'_'.$resourceName.'.'.$format;
    }

    /**
     * @param  array  $row
     * @return array
     */
    //    protected function makeHeaders(array $row) : array
    //    {
    //        $headers = [];
    //
    //        foreach ($row as $key => $value) {
    //            $value = $this->getDataValue($key);
    //
    //            if (! is_array($value)) {
    //                $headers[] = $key;
    //            } else {
    //                $keys = array_keys($value);
    //                array_walk($keys, function (&$currentKey) use ($key) {
    //                    $currentKey = $key.'.'.$currentKey;
    //                });
    //
    //                $updatedValue = [];
    //
    //                foreach ($keys as $key) {
    //                    if ($this->isWarehousesField($key)) {
    //                        $parts = explode('item_inventory.warehouses.', $key); // We use the warehouse name which may include a dot
    //                    } else {
    //                        $parts = explode('.', $key);
    //                    }
    //                    $index = end($parts);
    //                    if (isset($value[$index])) {
    //                        $updatedValue[$key] = $value[$index];
    //                    }
    //                }
    //
    //                $headers = array_merge(
    //          $headers,
    //          $this->makeHeaders($updatedValue)
    //        );
    //            }
    //        }
    //
    //        return $headers;
    //    }

    private function isWarehousesField(string $key): bool
    {
        return Str::contains($key, 'item_inventory.warehouses.');
    }

    /**
     * @return array|\ArrayAccess|mixed
     */
    protected function getDataValue($key)
    {
        // We get the first row with non-empty data for the key
        foreach ($this->data as $value) {
            $valueForKey = Arr::get($value, $key);
            if (! empty($valueForKey)) {
                return $valueForKey;
            }
        }

        // All of them are empty,
        // we get the value for the first row
        return Arr::get($this->data, $key);
    }

    public static function makeExportableField($name, bool $importable = true, ?string $label = ''): array
    {
        return [
            'exported_as' => $name,
            'importable' => $importable,
            'label' => ! empty($label) ? $label : ucwords(str_replace('.', ' ', str_replace('_', ' ', Str::title($name)))),
        ];
    }
}
