<?php

namespace App\DataTable;

use App\DTO\DataTableColumnGroupData;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Spatie\LaravelData\DataCollection;

trait DataTableModelTrait
{
    protected array $availableColumns;
    protected array $filterableColumns;

    public static function isDeletable(): bool
    {
        return false;
    }

    public static function frozenColumns(): array
    {
        return ['id'];
    }

    public static function defaultSort(): array
    {
        return [
            [
                'field' => 'id',
                'sort' => 'desc'
            ]
        ];
    }

    public static function defaultDensity(): string
    {
        return 'compact';
    }

    /*
     * The significance of available columns is it is what shows up in the table specifications columns, which drives
     * the frontend data table slots.  They don't necessarily have to line up with the resource.  This also determines
     * the order the columns appear in the data table.
     */
    public function availableColumns(): array
    {
        $this->availableColumns = static::getAllColumnsAndTypes();
        return $this->availableColumns;
    }

    public static function getAllColumnsAndTypes(): array
    {
        $tableName = app(static::class)->getTable();

        $databaseName = config('database.connections.mysql.database');

        $columns = Cache::remember("{$tableName}_columns", 1, function () use ($databaseName, $tableName) {
            return DB::select("
                SELECT COLUMN_NAME, DATA_TYPE 
                FROM information_schema.COLUMNS 
                WHERE TABLE_SCHEMA = ? 
                AND TABLE_NAME = ?
            ", [$databaseName, $tableName]);
        }) ;

        // Convert the result to an associative array
        $columnsAndTypes = [];
        foreach ($columns as $column) {
            $columnsAndTypes[$column->COLUMN_NAME] = $column->DATA_TYPE;
        }

        return $columnsAndTypes;
    }

    /*
     * Visible columns are what shows up as marked visible in the table specifications columns.  This drives what fields
     * appears as visible by default in the frontend data table.  By default, all available columns are visible.
     */
    public function visibleColumns(): array
    {
        return array_keys($this->availableColumns);
    }

    public static function specialLabels(): array
    {
        return [
            'id' => 'ID',
        ];
    }

    /*
     * Filterable columns are what shows up as marked filterable in the table specifications columns.  This drives what
     * appears on the filterable list in the frontend data table.  By default, all available columns are filterable.
     */
    public function filterableColumns(): array
    {
        $this->filterableColumns = array_keys($this->availableColumns ?? $this->availableColumns());
        return $this->filterableColumns;
    }

    public function scopeColumns(): array
    {
        return [];
    }

    public function generalFilterableColumns(): array
    {
        return ['id'];
    }

    public function sortableColumns(): array
    {
        return $this->filterableColumns ?? $this->filterableColumns();
    }

    public function columnGroups(): DataCollection
    {
        $groupings = [];
        foreach ($this->availableColumns() as $key => $type) {
            $groupings[] = DataTableColumnGroupData::from([
                'column' => $key,
                'group' => 'Basic Info',
                'group_data' => 'basic_info',
            ]);
        }
        return DataTableColumnGroupData::collection($groupings);
    }
}