<?php

namespace App\Data;

use App\Models\SalesCredit;
use App\Models\SalesOrderLine;
use Illuminate\Contracts\Support\Arrayable;
use Spatie\LaravelData\Attributes\DataCollectionOf;
use Spatie\LaravelData\Attributes\Validation\BooleanType;
use Spatie\LaravelData\Attributes\Validation\Exists;
use Spatie\LaravelData\Attributes\Validation\RequiredWithout;
use Spatie\LaravelData\Attributes\Validation\Rule;
use Spatie\LaravelData\Data;
use Spatie\LaravelData\DataCollection;
use Spatie\LaravelData\Optional;
use Spatie\LaravelData\Support\Validation\ValidationContext;
use Illuminate\Validation\Validator;

class SalesCreditData extends Data
{
    public function __construct(
        #[RequiredWithout('customer_id'),Exists('sales_orders','id')]
        public int|Optional $sales_order_id,

        #[Rule('max:255')]
        public string|Optional $sales_credit_number,

        #[Exists('warehouses','id')]
        public int|Optional $to_warehouse_id,
        public string|Optional $sales_credit_note,

        #[Rule('date')]
        public string|Optional $credit_date,

        #[Exists('stores','id')]
        public int|Optional $store_id,

        #[RequiredWithout('sales_order_id')]
        public int|Optional $customer_id,

        #[Exists('currencies','id')]
        public int|Optional $currency_id,

        #[Exists('currencies','code')]
        public string|Optional $currency_code,
        public array|Optional $tax_lines,

        #[Exists('tax_rates','id')]
        public int|Optional $tax_rate_id,

        #[DataCollectionOf(SalesCreditLineData::class)]
        public DataCollection $sales_credit_lines,

        public bool $is_tax_included = false,

        public bool $is_product = false,

    ) {
    }

    public static function withValidator(Validator $validator): void
    {
        if ($validator->passes()) {

            $validator->after(function (Validator $validator) {
                $attributes = $validator->attributes();
                $salesOrderId = $attributes['sales_order_id'] ?? null;

                // check sales order lines belong to the sales order and unreturned
                if (! empty($attributes['sales_credit_lines'])) {
                    $salesOrderLines = SalesOrderLine::with(['salesCreditLines', 'salesOrderFulfillmentLines'])
                        ->whereIn('id', array_column($attributes['sales_credit_lines'], 'sales_order_line_id'))
                        ->get();

                    foreach ($attributes['sales_credit_lines'] as $index => $creditLine) {
                        if (isset($creditLine['sales_order_line_id'])) {
                            /** @var SalesOrderLine $salesOrderLine */
                            $salesOrderLine = $salesOrderLines->firstWhere('id', $creditLine['sales_order_line_id']);

                            if ($salesOrderId) {
                                if ($salesOrderId && $salesOrderLine->sales_order_id != $salesOrderId) {
                                    $validator->errors()->add("sales_credit_lines.{$index}.sales_order_line_id",__('messages.sales_order.is_not_belongs'));
                                }
                            }

                            $creditedQuantity = $salesOrderLine->salesCreditLines->sum('quantity');
                            $remainingQuantity = $salesOrderLine->quantity - $creditedQuantity;

                            if ($creditLine['quantity'] > $remainingQuantity) {
                                $validator->errors()->add("sales_credit_lines.{$index}.quantity", __('messages.sales_order.greater_than_unreturned'));
                            }
                        }
                    }
                }
            });
        }
    }



    public static function messages(): array
    {
        return [
            'is_not_belongs_to_sales_order' => __('messages.sales_order.is_not_belongs'),
            'is_returned' => __('messages.sales_order.fully_returned_line'),
            'is_not_fulfilled' => __('messages.sales_order.is_not_fulfilled'),
            'is_greater_than_unreturned_quantity' => __('messages.sales_order.greater_than_unreturned'),
            'is_not_closed' => __('messages.sales_order.is_not_closed'),
        ];
    }
}