<?php

namespace App\Data;

use App\Models\Product;
use App\Models\Warehouse;
use Spatie\LaravelData\Attributes\Validation\Rule;
use Spatie\LaravelData\Data;
use Illuminate\Validation\Validator;
use Spatie\LaravelData\Optional;

class CreateInventoryAdjustmentData extends Data
{
    const ADJUSTMENT_TYPE_INCREASE = 'increase';
    const ADJUSTMENT_TYPE_DECREASE = 'decrease';
    const ADJUSTMENT_TYPE_SET = 'set';

    public function __construct(
        #[Rule('required|date')]
        public string $adjustment_date,
        #[Rule('required|exists:products,id')]
        public int|Optional $product_id,
        #[Rule('required|exists:warehouses,id')]
        public int $warehouse_id,
        #[Rule('required|numeric|lt:1000000')]
        public int $quantity,
        #[Rule('nullable|numeric|gte:0')]
        public float|Optional $unit_cost,
        public ?string $notes,
        public Optional|string $link_type,
        public Optional|int $link_id,
        #[Rule('required|in:'.self::ADJUSTMENT_TYPE_DECREASE.','.self::ADJUSTMENT_TYPE_DECREASE.','.self::ADJUSTMENT_TYPE_SET)]
        public string $adjustment_type = self::ADJUSTMENT_TYPE_INCREASE,
        #[Rule('nullable|exists:warehouse_locations,id')]
        public ?int $warehouse_location_id = null,
    ) {
    }

    public static function messages(): array
    {
        return [
            'warehouse_id.NeedsDefaultLocation' => __('messages.warehouse.warehouse_needs_location'),
            'quantity.InvalidQuantity' => __('messages.inventory.adjustment_quantity_not_zero'),
            'unit_cost.Required' => __('messages.unit_cost_required'),
        ];
    }

    public static function withValidator(Validator $validator): void
    {
        $validator->after(function ($validator) {
            $data = $validator->getData();
            if ($data['adjustment_type'] === self::ADJUSTMENT_TYPE_INCREASE && !isset($data['unit_cost'])) {
                $product = Product::find($data['product_id']);
                if (!$product || !$product->average_cost) {
                    $validator->errors()->add('unit_cost', 'The unit cost is required for an increase adjustment if the product does not have an average cost.');
                }
            }

            if (in_array($data['adjustment_type'], [self::ADJUSTMENT_TYPE_INCREASE, self::ADJUSTMENT_TYPE_SET]) && empty($data['unit_cost'])) {
                $validator->errors()->add('unit_cost', 'The unit cost is required.');
            }

            if (in_array($data['adjustment_type'], [self::ADJUSTMENT_TYPE_DECREASE, self::ADJUSTMENT_TYPE_INCREASE]) && $data['quantity'] == 0) {
                $validator->errors()->add('quantity', 'The quantity must be non-zero for increase/decrease adjustments.');
            }

            $warehouse = Warehouse::find($data['warehouse_id']);
            if (!$warehouse || !$warehouse->defaultLocation) {
                $validator->errors()->add('warehouse_id', 'The warehouse must have a default location.');
            }
        });
    }
}
