<?php

namespace App\Console\Patches\Shopify;

use App\Models\SalesOrderFulfillment;
use App\Models\Shopify\ShopifyOrder;
use App\Models\Shopify\ShopifyOrderMapping;
use App\Repositories\Shopify\ShopifyOrderMappingRepository;
use Illuminate\Console\Command;

class PortShopifyOrderMappingToDedicatedModel extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'sku:shopify:port-order-mappings';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Ports shopify order mappings for refunds and fulfillments into the new model.';

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        $this->info('Porting refund maps...');
        $this->portMappings('refundsMap', [$this, 'portRefundsForOrder']);
        $this->info('Refund maps ported.');
        $this->info('Porting fulfillment maps...');
        $this->portMappings('fulfillmentsMap', [$this, 'portFulfillmentsForOrder']);
        $this->info('Fulfillment maps ported.');

        return 0;
    }

    private function portMappings(string $key, callable $orderHandler)
    {
        ShopifyOrder::with([])
            ->whereNotNull($key)
            ->where($key, '!=', '[]')
            ->chunk(10000, function ($orders) use ($orderHandler) {
                foreach ($orders as $shopifyOrder) {
                    $this->info("Porting order: $shopifyOrder->name...");
                    $orderHandler($shopifyOrder);
                    $this->info("Order: $shopifyOrder->name ported.");
                }
            });
    }

    private function portFulfillmentsForOrder(ShopifyOrder $order)
    {
        $fulfillmentsMap = collect($order->fulfillmentsMap ?: []);
        $fulfillmentsMap->each(function ($map, $index) use ($order, $fulfillmentsMap) {
            if (! is_array($map)) {
                return;
            }
            $fulfillmentId = $map['shopify_fulfillment_id'];
            $fulfillment = collect($order->fulfillments ?: [])
                ->where('id', $fulfillmentId)
                ->first();

            if (! $fulfillment) {
                // skip
                return;
            }

            foreach ($fulfillment['line_items'] as $fulfillmentLineItem) {
                app(ShopifyOrderMappingRepository::class)->create([
                    'shopify_order_id' => $order->id,
                    'line_id' => $fulfillmentLineItem['id'],
                    'line_type' => ShopifyOrderMapping::LINE_TYPE_LINE_ITEM,
                    'quantity' => $fulfillmentLineItem['quantity'],
                    'processed_at' => $fulfillment['created_at'],
                    'no_restock_fulfillments_applied' => null,
                    'link_id' => $fulfillment['id'],
                    'link_type' => ShopifyOrderMapping::LINK_TYPE_FULFILLMENTS,
                    'sku_link_id' => $map['sku_fulfillment_id'],
                    'sku_link_type' => SalesOrderFulfillment::class,
                ]);
            }
            $fulfillmentsMap->offsetUnset($index);
        });

        if ($fulfillmentsMap->isEmpty()) {
            $this->info('fixed');
            $order->fulfillmentsMap = null;
        } else {
            $this->info('not fixed');
            $order->fulfillmentsMap = $fulfillmentsMap->toArray();
        }
        $order->save();
    }

    private function portRefundsForOrder(ShopifyOrder $order)
    {
        $refundsMap = collect($order->refundsMap ?: []);
        $refundsMap->each(function ($map, $refundId) use ($order, $refundsMap) {
            $refund = collect($order->refunds ?: [])
                ->where('id', $refundId)
                ->first();

            if (! $refund) {
                // skip
                return;
            }

            $order->markRefundAsProcessed($refundId, $map['sku_link_id'], $map['sku_link_type']);

            $refundsMap->offsetUnset($refundId);
        });

        if ($refundsMap->isEmpty()) {
            $order->refundsMap = null;
        } else {
            $order->refundsMap = $refundsMap->toArray();
        }
        $order->save();
    }
}
