<?php

namespace App\Console\Patches\Shopify\Orders\Patches;

use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\Shopify\ShopifyOrder;
use App\Repositories\Shopify\ShopifyOrderMappingRepository;
use Carbon\Carbon;
use Illuminate\Console\Command;
use Illuminate\Support\Collection;

class MapFulfillments extends Command
{
    private ShopifyOrderMappingRepository $shopifyOrderMappingRepository;

    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'sku:shopify:orders:patch:map-fulfillments';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Map Shopify fulfillments to SKU fulfillments';

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
        $this->shopifyOrderMappingRepository = app(ShopifyOrderMappingRepository::class);
    }

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        $query = ShopifyOrder::with(['salesOrder.salesOrderFulfillments.salesOrderFulfillmentLines.salesOrderLine', 'orderMappings', 'integrationInstance'])
            ->whereRelation('salesOrder', 'order_status', '!=', SalesOrder::STATUS_DRAFT)
            ->whereRaw("JSON_LENGTH(`json_object`, '$.fulfillments') > 0");

        $bar = $this->output->createProgressBar($query->count());
        $bar->start();

        $query->each(function (ShopifyOrder $order) use ($bar) {
            collect($order->fulfillments)
                ->where('status', 'success')->values()
                ->each(function (array $fulfillment, $index) use ($bar, $order) {
                    // reload orderMappings and sku fulfillments if the order has many fulfillments
                    if ($index != 0) {
                        $order->refresh();
                    }

                    // pre audit trail
                    if ($order->integrationInstance->audit_trail_start_date->gt(Carbon::parse($fulfillment['created_at']))) {
                        $bar->advance();

                        return;
                    }

                    // the fulfillment is mapped
                    $mapped = $this->shopifyOrderMappingRepository->shopifyOrderIsMappedToFulfillmentId($order, $fulfillment['id']);

                    if (! $mapped) {
                        $mapped = $this->mapShopifyFulfillment($order, $fulfillment);
                        if ($mapped) {
                            $this->info("\nFulfillment {$fulfillment['name']}(ID: {$fulfillment['id']}) Mapped");
                        } else {
                            $this->warn("\nCan't map fulfillment {$fulfillment['name']}(ID: {$fulfillment['id']}) to SKU fulfillment");
                        }
                    }

                    $bar->advance();
                });
        });

        return self::SUCCESS;
    }

    private function mapShopifyFulfillment(ShopifyOrder $order, array $fulfillment): bool
    {
        $mappedSkuFulfillmentIds = $this->shopifyOrderMappingRepository->getFulfillmentIdsForShopifyOrder($order);

        $unmappedSkuFulfillments = $order->salesOrder->salesOrderFulfillments->whereNotIn('id', $mappedSkuFulfillmentIds);

        // we will try to map by lines
        $salesOrderFulfillment = $this->getSkuFulfillmentByLines($fulfillment, $unmappedSkuFulfillments);
        if ($salesOrderFulfillment) {
            $order->mapFulfillmentToSkuFulfillment($fulfillment, $salesOrderFulfillment->id);

            return true;
        }

        return false;
    }

    private function getSkuFulfillmentByLines(array $fulfillment, Collection $unmappedSkuFulfillments): ?SalesOrderFulfillment
    {
        $shopifyLineItems = collect($fulfillment['line_items'])->pluck('quantity', 'id')->toArray();

        return $unmappedSkuFulfillments->filter(function (SalesOrderFulfillment $salesOrderFulfillment) use ($shopifyLineItems) {
            $skuLines = $salesOrderFulfillment->salesOrderFulfillmentLines->pluck('quantity', 'salesOrderLine.sales_channel_line_id')->toArray();
            $skuLinesCount = count($skuLines);

            // same lines and quantities
            if (count($shopifyLineItems) == $skuLinesCount && count(array_intersect_assoc($shopifyLineItems, $skuLines)) == $skuLinesCount) {
                return true;
            }

            return false;
        })->first();
    }
}
