<?php

namespace App\Console\Patches;

use App\Models\FifoLayer;
use App\Models\InventoryMovement;
use App\Models\PurchaseOrderLine;
use App\Models\PurchaseOrderShipmentReceiptLine;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;

class PortPOLineToShipmentReceiptLineInFifoLayers extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'sku:po-lines-to-receipt-lines';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Ports PO receipt lines to shipment receipt lines in fifo layers.';

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        set_time_limit(0);

        $withIssues = [];
        $totalFixed = 0;

        $query = FifoLayer::with(['inventoryMovements'])
            ->where('link_type', PurchaseOrderLine::class);
        $totalAffected = $query->count();
        $query->each(function (FifoLayer $fifoLayer) use (&$withIssues, &$totalFixed) {
            /**
             * We get the inventory movement that created
             * the fifo layer and map the fifo layer link
             * to the attached purchase receipt.
             */
            /** @var InventoryMovement|null $movement */
            $movement = $fifoLayer->inventoryMovements()
                ->where('quantity', '>=', 0)
                ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
                ->where('link_type', PurchaseOrderShipmentReceiptLine::class)
                ->where('type', InventoryMovement::TYPE_PURCHASE_RECEIPT)
                ->first();

            if ($movement) {
                $fifoLayer->link_type = PurchaseOrderShipmentReceiptLine::class;
                $fifoLayer->link_id = $movement->link_id;
                $fifoLayer->save();
                $totalFixed += 1;
            } else {
                $withIssues[] = $fifoLayer->id;
            }

            // Next, we attempt to fix fifos without positive movement
            $this->fixPositiveMovement($fifoLayer);
        });

        if (! empty($withIssues)) {
            $this->output->error("Some receipt fifo layers don't have matching receipt movements: \n");
            $this->output->info($withIssues);

            return 1;
        }

        $this->output->info("$totalFixed / $totalAffected fixed.\n");

        return 0;
    }

    private function fixPositiveMovement(FifoLayer $fifoLayer)
    {
        $positiveMovement = $fifoLayer->inventoryMovements()
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
            ->where('quantity', '>', 0)
            ->whereHasMorph('link', [PurchaseOrderLine::class, PurchaseOrderShipmentReceiptLine::class])
            ->first();
        if ($positiveMovement) {
            return;
        }

        $link = $fifoLayer->link;

        $receipt = null;
        if ($link instanceof PurchaseOrderLine) {
            $receipt = $link->purchaseOrderShipmentReceiptLines()
                ->where('purchase_order_shipment_receipt_lines.quantity', $link->received_quantity)
                ->first();
        } elseif ($link instanceof PurchaseOrderShipmentReceiptLine) {
            $receipt = $link;
        }

        if ($receipt) {
            DB::transaction(function () use ($fifoLayer, $receipt, $link) {
                $movement = new InventoryMovement();
                $movement->inventory_movement_date = $receipt->purchaseOrderShipmentReceipt->received_at ?? now();
                $movement->product_id = $fifoLayer->product_id;
                $movement->quantity = $fifoLayer->original_quantity;
                $movement->type = InventoryMovement::TYPE_PURCHASE_RECEIPT;
                $movement->warehouse_id = $fifoLayer->warehouse_id;
                $movement->warehouse_location_id = $fifoLayer->warehouse->defaultLocation?->id ?? null;
                $movement->link_id = $receipt->id;
                $movement->link_type = PurchaseOrderShipmentReceiptLine::class;
                $movement->reference = $receipt->purchaseOrderShipmentLine->purchaseOrderLine->purchaseOrder->purchase_order_number;
                $fifoLayer->inventoryMovements()->save($movement);

                if ($link instanceof PurchaseOrderLine) {
                    $fifoLayer->link_id = $receipt->id;
                    $fifoLayer->link_type = PurchaseOrderShipmentReceiptLine::class;
                    $fifoLayer->save();
                }
            });
        }
    }
}
