<?php

namespace App\Console\Patches;

use App\Models\Shopify\ShopifyOrder;
use Illuminate\Console\Command;
use Illuminate\Database\QueryException;
use Illuminate\Support\Facades\Log;

class FixShopifyToSkuOrderLink extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'sku:shopify:fix-order-links 
                                {--i|integrationInstance= : The ID of the integration instance}
                                {--r|recreate : Whether to attempt to recreate the broken sku order }';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Fixes broken sku order links for shopify orders.';

    /**
     * Execute the console command.
     */
    public function handle(): void
    {
        Log::info('Running shopify broken orders fix');

        $integrationInstance = (int) $this->option('integrationInstance') ?? null;
        $recreate = $this->option('recreate') ?? false;

        $query = ShopifyOrder::query();
        $query = $query->whereNotNull('sku_sales_order_id');

        if ($integrationInstance) {
            $query = $query->where('integration_instance_id', $integrationInstance);
        }

        $query = $query->whereDoesntHave('salesOrder');
        $affected = $query->count();
        $query->each(function (ShopifyOrder $order) use ($recreate) {
            $this->output->text("Fixing link for order: {$order->order_number}.\n");

            // Reset the link
            $order->sku_sales_order_id = null;
            $order->save();

            $this->output->text("Link fixed for order: {$order->order_number}.\n");

            // We attempt to recreate the sku order if needed.
            if ($recreate) {
                $this->output->text("Creating sku order for: {$order->order_number}.\n");
                try {
                    $order->createSKUOrder();
                    $this->output->text("SKU order created for: {$order->order_number}.\n");
                } catch (QueryException $queryException) {
                    // Duplicate entry
                    if (($queryException->errorInfo[0] ?? 0) == 23000) {
                        $this->warn("SKU order already created for: {$order->order_number}.");
                    }
                    $this->error("{$order->order_number} Exception: {$queryException->getMessage()}");
                }
            }
        });

        $this->output->text("Links fixed for {$affected} orders(s).");
        Log::info("Links fixed for {$affected} orders(s).");
    }
}
