<?php

namespace App\Console\Patches;

use App\Helpers;
use App\Jobs\UpdateProductsInventoryAndAvgCost;
use App\Models\SalesCredit;
use App\Models\SalesOrder;
use App\Models\Setting;
use App\Models\Shopify\ShopifyOrder;
use App\Notifications\MonitoringMessage;
use Carbon\Carbon;
use Facades\App\Services\Shopify\Orders\Actions\ShopifyDownloadOrder;
use Generator;
use Illuminate\Console\Command;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Notification;
use Symfony\Component\Console\Helper\ProgressBar;
use Throwable;

class FixLoopReturnsData extends Command
{
    const ORDERS_PER_PAGE = 500;

    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'sku:fix-loop-returns-data
                            {--o|orders=* : The orders to fix data for.}
                            {--debug : Run in debug mode }';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Fixes orders that are canceled in SKU even though they have no_restock type in shopify.';

    protected Carbon $skipAdjustmentsStartDate;

    protected Carbon $maxApplicableOrdersDate;

    protected int $affectedOrders;

    protected int $processedOrders = 0;

    protected ProgressBar $progressBar;

    public function __construct()
    {
        parent::__construct();

        $tz = Helpers::setting(Setting::KEY_DEFAULT_TIMEZONE);

        $this->skipAdjustmentsStartDate = Carbon::parse(
            // We should skip orders with lines having adjustments on or after March 1, 2022.
            // @see https://siberventures.atlassian.net/browse/SKU-4693
            '2022-03-01',
            $tz
        );

        $this->maxApplicableOrdersDate = Carbon::parse(
            // Not processing orders after the fix release date.
            '2022-05-04',
            $tz
        );
    }

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        set_time_limit(0);

        foreach ($this->getOrders() as $orders) {
            try {
                $this->fixOrders($orders);
            } catch (Throwable $e) {
                $this->error("\nError: {$e->getMessage()}");
                try {
                    Notification::route('slack', config('slack.debugging'))->notify(new MonitoringMessage("Loop Data Fix: {$e->getMessage()}, {$e->getFile()}:{$e->getLine()}"));
                } catch (Throwable) {
                }
            } finally {
                //            $this->progressBar->finish();
            }
        }

        return 0;
    }

    private function getOrders(): Generator
    {
        if (! empty($this->option('orders'))) {
            yield explode(',', $this->option('orders')[0]);

            return;
        }

        $this->affectedOrders = ShopifyOrder::with([])
            ->whereRaw(
                DB::raw('json_extract(refunds, "$[*].refund_line_items[*].restock_type") LIKE "%no_restock%"')->getValue(DB::getQueryGrammar())
            )
            ->whereDate('sku_created_at', '<=', $this->maxApplicableOrdersDate)
            ->whereHas('salesOrder', function (Builder $builder) {
                return $builder->where('order_status', '!=', SalesOrder::STATUS_DRAFT)
                    ->whereDate('updated_at', '<=', $this->maxApplicableOrdersDate)
                    ->whereHas('salesOrderLines', function (Builder $builder) {
                        return $builder->where('no_audit_trail', 0)
                            ->where('canceled_quantity', '>', 0);
                    });
            })
            ->count();

        $this->warn("$this->affectedOrders orders affected.");
        if (! $this->confirm('Start processing?')) {
            yield [];

            return;
        }

        //        $this->progressBar = $this->output->createProgressBar($this->affectedOrders);
        //        $this->progressBar->start();

        $page = 1;

        do {
            $results = ShopifyOrder::with([])
                ->whereRaw(
                    DB::raw('json_extract(refunds, "$[*].refund_line_items[*].restock_type") LIKE "%no_restock%"')->getValue(DB::getQueryGrammar())
                )
                ->whereDate('sku_created_at', '<=', $this->maxApplicableOrdersDate)
                ->whereHas('salesOrder', function (Builder $builder) {
                    return $builder->where('order_status', '!=', SalesOrder::STATUS_DRAFT)
                        ->whereDate('updated_at', '<=', $this->maxApplicableOrdersDate)
                        ->whereHas('salesOrderLines', function (Builder $builder) {
                            return $builder->where('no_audit_trail', 0)
                                ->where('canceled_quantity', '>', 0);
                        });
                })
                ->forPage($page, self::ORDERS_PER_PAGE)
                ->pluck('name')
                ->toArray();

            yield $results;

            $page++;
        } while (count($results) === self::ORDERS_PER_PAGE);
    }

    /**
     * @throws Throwable
     */
    private function fixOrders(array $orders): void
    {
        foreach ($orders as $orderNumber) {
            $order = $this->findOrderByNumber($orderNumber);

            /*$order->salesOrderFulfillments()
                ->each(function (SalesOrderFulfillment $fulfillment) {
                    // First, we attempt to delete the order in starshipit.
                    try {
                        $fulfillment->delete(true);

                        $this->info("Deleted $fulfillment->fulfillment_sequence.");
                    } catch (\Exception $e) {
                        // We ignore
                        $this->info($e->getMessage());
                        $this->info("Skipping...");
                    }
                });
            */
            // This is only available for Shopify Orders.
            if (! $order->shopifyOrder) {
                $this->error("\nSkipping non-shopify order: $order->sales_order_number");
                $this->incrementProcessed();

                continue;
            }

            // We skip draft orders
            if ($order->isDraft()) {
                $this->warn("\nSkipping draft order: {$order->sales_order_number}");
                $this->incrementProcessed();

                continue;
            }

            // We skip orders without audit trail lines and
            // orders without canceled lines as those are what
            // may be affected by the loop returns data.
            if (! $this->orderHasAuditTrailLines($order) || ! $this->orderHasCanceledLines($order)) {
                $this->incrementProcessed();

                continue;
            }

            // Skip orders that exceed the max date for affected orders.
            if ($order->created_at->gt($this->maxApplicableOrdersDate)) {
                $this->warn("\nSkipping order: $order->sales_order_number as it was after the max incident date.");
                $this->incrementProcessed();

                continue;
            }

            // Skip orders with adjustments in the range of dates
            // where the user attempted to fix the data for.
            if ($this->shouldSkipDueToAdjustment($order)) {
                $this->warn("\nSkipping order: $order->sales_order_number due to existing adjustments.");
                $this->incrementProcessed();

                continue;
            }

            DB::beginTransaction();

            try {
                // To prepare the order for a refresh, we clear any canceled quantities on the sales order line.
                // Also, we remove any sales credit that may exist on the sales order.
                $this->warn("\nRefreshing order: $order->sales_order_number...");

                if ($this->debugging() && ! $this->confirm('Continue? y/n')) {
                    $this->warn("\nSkipping order: $order->sales_order_number.");

                    return;
                }

                $this->refreshOrder(
                    $this->prepareOrderForRefresh($order)
                );

                DB::commit();

                $this->info("\nOrder: $order->sales_order_number REFRESHED!");
            } catch (Throwable $e) {
                DB::rollBack();
                throw $e;
            } finally {
                $this->incrementProcessed();
                //$this->info("\n$this->processedOrders/$this->affectedOrders");
            }
        }
    }

    private function incrementProcessed()
    {
        $this->processedOrders++;
    }

    private function debugging(): bool
    {
        return ! empty($this->option('debug'));
    }

    private function refreshOrder(SalesOrder $salesOrder): void
    {
        ShopifyDownloadOrderrefresh($salesOrder->shopifyOrder);

        $query = $salesOrder->salesOrderLines()
            ->where('is_product', true)
            ->whereNotNull('product_id')
            ->whereNotNull('warehouse_id')
            ->where('no_audit_trail', 0);

        if ($query->count() > 0) {
            UpdateProductsInventoryAndAvgCost::dispatch(
                $query
                    ->pluck('product_id')
                    ->toArray()
            );
        }
    }

    private function prepareOrderForRefresh(SalesOrder $salesOrder): SalesOrder
    {
        // We open the sales order to allow for the refresh.
        if (! $salesOrder->isOpen()) {
            $salesOrder = $salesOrder->open();
        }

        // We clear any canceled quantities on the sales order lines
        // that don't have inventory movements.
        $salesOrder->salesOrderLines()
            ->whereDoesntHave('inventoryMovements')
            ->where('canceled_quantity', '>', 0)
            ->where('no_audit_trail', 0)
            ->update([
                'quantity' => DB::raw('`quantity` + `canceled_quantity`'),
                'canceled_quantity' => 0,
            ]);

        // We delete any sales credit attached to the order.
        $salesOrder->salesCredits->each(function (SalesCredit $salesCredit) {
            $salesCredit->delete();
        });

        // We clear the canceled_at date.
        if ($salesOrder->canceled_at) {
            $salesOrder->canceled_at = null;
            $salesOrder->save();
        }

        // Finally, we clear the refund map on the shopify order.
        $salesOrder->shopifyOrder->clearRefundMappings();
        $salesOrder->shopifyOrder->save();

        return $salesOrder->refresh();
    }

    private function findOrderByNumber(string $salesOrderNumber): SalesOrder|Model
    {
        return SalesOrder::with(['salesOrderLines'])
            ->where('sales_order_number', $salesOrderNumber)
            ->firstOrFail();
    }

    private function shouldSkipDueToAdjustment(SalesOrder $salesOrder): bool
    {
        // Disabling this check as adjustments will be deleted.
        return false;
        //        return $salesOrder
        //            ->salesOrderLines()
        //            ->with(['product', 'product.inventoryAdjustments'])
        //            ->whereHas('product.inventoryAdjustments', function (Builder $builder) {
        //                return $builder->whereDate('adjustment_date', '>=', $this->skipAdjustmentsStartDate);
        //            })->count() > 0;
    }

    private function orderHasAuditTrailLines(Model|SalesOrder $order): bool
    {
        return $order->salesOrderLines()
            ->where('no_audit_trail', 0)
            ->count() > 0;
    }

    private function orderHasCanceledLines(Model|SalesOrder $order): bool
    {
        return $order->salesOrderLines()
            ->where('canceled_quantity', '>', 0)
            ->count() > 0;
    }
}
