<?php

namespace App\Console\Patches;

use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use Illuminate\Console\Command;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\DB;

class FixFulfilledOrdersWithBackorders extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'sku:orders:reverse-invalid-fulfillments';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Reverses fulfillments for fulfilled sales orders with active backorder queues.';

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        $deleteCount = 0;

        $query = SalesOrderFulfillment::with([
            'salesOrder',
            'salesOrderFulfillmentLines',
        ])
            ->whereHas('salesOrder', function (Builder $builder) {
                return $builder//->where('order_status', SalesOrder::STATUS_CLOSED)
                    ->whereHas('salesOrderLines', function (Builder $builder) {
                        return $builder->
                      whereHas('backorderQueue', function (Builder $builder) {
                          return $builder->active();
                      });
                    });
            });

        $affected = $query->count();
        $query
            ->cursor()
            ->each(function (SalesOrderFulfillment $fulfillment) use (&$deleteCount, $affected) {
                try {
                    DB::transaction(function () use ($fulfillment) {
                        $fulfillment->delete();
                    });
                    $deleteCount++;
                    $this->info("Deleted {$deleteCount} of {$affected}");
                } catch (\Exception $e) {
                    $this->output->error("Fulfillment {$fulfillment->id} failed: {$e->getMessage()}, \n {$e->getFile()}: {$e->getLine()}");
                }
            });

        $this->info('Total of '.$deleteCount.' invalid fulfillments reversed.');

        // Refresh product inventory
        if ($deleteCount > 0) {
            Artisan::call('sku:inventory:refresh');
        }
    }
}
