<?php

namespace App\Console;

use App\Console\Commands\Inventory\ForecastInventoryCommand;
use App\Console\Commands\ReleaseBackorderQueues;
use App\Console\Commands\RemoveImportExportFiles;
use App\Console\Commands\Shopify\ShopifySubmitFulfillments;
use App\Enums\DownloadedBy;
use App\Enums\FinancialAlertNotificationConditionEnum;
use App\Helpers;
use App\Jobs\BackorderPurchasingJob;
use App\Jobs\GenerateCacheProductListingQuantityJob;
use App\Jobs\InventorySnapshotJob;
use App\Jobs\ProcessNewInventory;
use App\Jobs\PurchaseOrderSubmissionBatchingJob;
use App\Jobs\SendFinancialAlertSummaryJob;
use App\Jobs\ShipStation\GetShipments;
use App\Jobs\Shopify\GenerateCreateSalesOrderFromShopifyOrderJobJobs;
use App\Jobs\Shopify\GenerateUpdateSalesOrderFromShopifyOrderJobJobs;
use App\Jobs\Shopify\ShopifyDeleteMissingProductVariantsJob;
use App\Jobs\UpdateSMOTrackingInfo;
use App\Models\Integration;
use App\Models\IntegrationInstance;
use App\Models\Setting;
use App\Models\Supplier;
use App\Repositories\IntegrationInstanceRepository;
use App\Repositories\NominalCodeRepository;
use App\Repositories\SupplierRepository;
use Carbon\Carbon;
use Exception;
use Illuminate\Console\Scheduling\Schedule;
use Illuminate\Contracts\Events\Dispatcher;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Foundation\Console\Kernel as ConsoleKernel;
use Illuminate\Support\Facades\Artisan;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Enums\Entities\AmazonReportTypeEnum;
use Modules\Amazon\Jobs\AmazonProcessUnprocessedNewInboundShipmentsJob;
use Modules\Amazon\Jobs\AmazonProcessUnprocessedRemovalOrdersJob;
use Modules\Amazon\Repositories\AmazonIntegrationInstanceRepository;
use Modules\Ebay\ApiDataTransferObjects\EbayGetLegacyProductsAdt;
use Modules\Ebay\ApiDataTransferObjects\EbayGetOrdersAdt;
use Modules\Ebay\Entities\EbayIntegrationInstance;
use Modules\Ebay\Jobs\CreateSkuOrdersFromEbayOrdersJob;
use Modules\Ebay\Jobs\RefreshEbayLegacyProductsJob;
use Modules\Ebay\Jobs\RefreshEbayOrdersJob;
use Modules\Qbo\Entities\QboIntegrationInstance;
use Modules\ShipMyOrders\Entities\ShipMyOrdersIntegrationInstance;
use Modules\ShipMyOrders\Jobs\ShipMyOrdersRefreshInventoryJob;
use Modules\Veracore\Entities\VeracoreIntegrationInstance;
use Modules\Veracore\Jobs\VeracoreUpdateTrackingJob;
use Modules\WooCommerce\ApiDataTransferObjects\WooCommerceGetOrdersAdt;
use Modules\WooCommerce\ApiDataTransferObjects\WooCommerceGetProductsAdt;
use Modules\WooCommerce\Entities\WooCommerceIntegrationInstance;
use Modules\WooCommerce\Jobs\CreateSkuOrdersFromWooCommerceJob;
use Modules\WooCommerce\Jobs\RefreshWooCommerceOrdersJob;
use Modules\WooCommerce\Jobs\RefreshWooCommerceProductsJob;
use Modules\Xero\Entities\XeroIntegrationInstance;

class Kernel extends ConsoleKernel
{
    private AmazonIntegrationInstanceRepository $amazonIntegrationInstanceRepository;

    private IntegrationInstanceRepository $integrationInstanceRepository;

    public function __construct(Application $app, Dispatcher $events)
    {
        $this->amazonIntegrationInstanceRepository = app(AmazonIntegrationInstanceRepository::class);
        $this->integrationInstanceRepository = app(IntegrationInstanceRepository::class);
        parent::__construct($app, $events);
    }

    /**
     * Define the application's command schedule.
     *
     *
     * @throws Exception
     */
    protected function schedule(Schedule $schedule): void
    {
        $events = [];

        /*
        |--------------------------------------------------------------------------
        | Accounting
        |--------------------------------------------------------------------------
        */

        if (app(NominalCodeRepository::class)->isAccountingNominalCodeMappingsSet() && Helpers::setting(Setting::KEY_ACCOUNTING_ENABLED)) {
            $events[] = $schedule->command('sku:accounting:transactions:sync')->everyFiveMinutes()->withoutOverlapping()->runInBackground();
        }

        /** @var IntegrationInstance $accountingInstance */
        if (($accountingInstance = $this->integrationInstanceRepository->getAccountingInstance()) && $accountingInstance->is_automatic_sync_enabled) {
            if ($accountingInstance instanceof XeroIntegrationInstance && $accountingInstance->hasRemainingDailyApiCalls()) {
                $events[] = $schedule->command('sku:xero:transactions:sync')->everyFiveMinutes()->withoutOverlapping()->runInBackground();
                $events[] = $schedule->command('sku:xero:payments:sync')->everyFiveMinutes()->withoutOverlapping()->runInBackground();
            }

            if ($accountingInstance instanceof QboIntegrationInstance && $accountingInstance->hasRemainingDailyApiCalls()) {
                $events[] = $schedule->command('qbo:accounts:refresh')->everyThirtyMinutes()->withoutOverlapping()->runInBackground();
                $events[] = $schedule->command('qbo:tax-rates:refresh')->everyThirtyMinutes()->withoutOverlapping()->runInBackground();
                $events[] = $schedule->command('qbo:tax-codes:refresh')->everyThirtyMinutes()->withoutOverlapping()->runInBackground();

                if($accountingInstance->is_automatic_sync_enabled){
                    $events[] = $schedule->command('qbo:sync:invoices')->everyFiveMinutes()->withoutOverlapping()->runInBackground();
                    $events[] = $schedule->command('qbo:sync:purchase-orders')->everyFiveMinutes()->withoutOverlapping()->runInBackground();
                    $events[] = $schedule->command('qbo:sync:invoice-payments')->everyFiveMinutes()->withoutOverlapping()->runInBackground();
                    $events[] = $schedule->command('qbo:sync:bills')->everyFiveMinutes()->withoutOverlapping()->runInBackground();
                    $events[] = $schedule->command('qbo:sync:journals')->everyFiveMinutes()->withoutOverlapping()->runInBackground();
                    $events[] = $schedule->command('qbo:sync:credit-memos')->everyFiveMinutes()->withoutOverlapping()->runInBackground();
                }
                
            }
        }

        /*
        |--------------------------------------------------------------------------
        | Reporting
        |--------------------------------------------------------------------------
        */

        //Financial Reporting
        $events[] = $schedule->command('sku:financials:calculate')
            ->everyFiveMinutes()->withoutOverlapping()->runInBackground()
            ->name('Calculate Proforma Financials');
        $events[] = $schedule->command('sku:financials:daily:calculate')
            ->everyFiveMinutes()->withoutOverlapping()->runInBackground()
            ->name('Calculate Daily Financials');

        // Invalidate cache for daily average consumption at the end of each local day
        $events[] = $schedule->command('sku:products:invalidate-daily-average-consumption')
            ->dailyAt(Helpers::utcStartOfLocalDate(Carbon::now())->format('H:i:s'))
            ->name('Invalidate Daily Average Consumption for Products at Midnight Tenant Timezone');

        // take a daily snapshot of product inventories
        $events[] = $schedule->job(InventorySnapshotJob::class)->daily();

        // send financial alert notifications
        $events[] = $schedule->job(new SendFinancialAlertSummaryJob(FinancialAlertNotificationConditionEnum::DAILY_SUMMARY))->daily();
        $events[] = $schedule->job(new SendFinancialAlertSummaryJob(FinancialAlertNotificationConditionEnum::WEEKLY_SUMMARY))->weekly();

        /*
        |--------------------------------------------------------------------------
        | Sales Channels
        |--------------------------------------------------------------------------
        */

        //Sync Inventory across sales channels
        if (config('app.env') == 'production') {
            IntegrationInstance::query()
                ->where('is_automatic_sync_enabled', true)
                ->whereHas('integration', function (Builder $query) {
                    $query->where('integration_type', Integration::TYPE_SALES_CHANNEL);
                })->each(function (IntegrationInstance $integrationInstance) use ($schedule, &$events) {
                    $events[] = $schedule->command('sku:sync-inventory -i '.$integrationInstance->id)
                        ->withoutOverlapping()
                        ->everyFifteenMinutes();
                });
        }

        if (app(AmazonIntegrationInstanceRepository::class)->autoSyncExists()) {
            if (AmazonIntegrationInstance::query()->exists()) {
                $events[] = $schedule
                    ->command('amazon:orders:reset-pii-data')
                    ->daily()
                    ->withoutOverlapping()
                    ->runInBackground()
                    ->name('Amazon PII Data Reset');
            }

            $this->amazonIntegrationInstanceRepository->allAutoSync()->each(function (AmazonIntegrationInstance $amazonIntegrationInstance) use ($schedule, &$events) {
                $events[] = $schedule
                    ->command('amazon:reports:create '.$amazonIntegrationInstance->id)
                    ->everyMinute()
                    ->withoutOverlapping()
                    ->runInBackground()
                    ->name('Create Amazon Reports');

                $events[] = $schedule
                    ->command('amazon:reports:request '.$amazonIntegrationInstance->id.' '.AmazonReportTypeEnum::PRODUCTS())
                    ->daily()
                    ->withoutOverlapping()
                    ->runInBackground()
                    ->name('Refresh Products for '.$amazonIntegrationInstance->name);

                $events[] = $schedule
                    ->command('amazon:orders:fulfill-mfn '.$amazonIntegrationInstance->id)
                    ->everyFifteenMinutes()
                    ->withoutOverlapping()
                    ->runInBackground()
                    ->name('Fulfill Amazon MFN Orders for '.$amazonIntegrationInstance->name);

                $events[] = $schedule
                    ->command('amazon:feeds:create '.$amazonIntegrationInstance->id)
                    ->everyFiveMinutes()
                    ->withoutOverlapping()
                    ->runInBackground()
                    ->name('Create Amazon Feeds for '.$amazonIntegrationInstance->name);

                $events[] = $schedule
                    ->command('amazon:feeds:refresh '.$amazonIntegrationInstance->id)
                    ->everyFifteenMinutes()
                    ->withoutOverlapping()
                    ->runInBackground()
                    ->name('Refresh Amazon Feeds for '.$amazonIntegrationInstance->name);

                if (@$amazonIntegrationInstance->integration_settings['is_fba_enabled']) {
                    $frequency = config('app.env') == 'production' ? '5 * * * *' : '0 1 * * *';

                    $events[] = $schedule
                        ->command('amazon:reports:request '.$amazonIntegrationInstance->id.' '.AmazonReportTypeEnum::FBA_INVENTORY())
                        ->cron($frequency)
                        ->withoutOverlapping()
                        ->runInBackground()
                        ->name('Refresh FBA Inventory for '.$amazonIntegrationInstance->name);

                    $events[] = $schedule
                        ->command('amazon:reports:request '.$amazonIntegrationInstance->id.' '.AmazonReportTypeEnum::FBA_REPORT_INVENTORY_LEDGER())
                        ->cron($frequency)
                        ->withoutOverlapping()
                        ->runInBackground()
                        ->name('Refresh FBA Inventory Ledger for '.$amazonIntegrationInstance->name);

                    $events[] = $schedule
                        ->command('amazon:reports:request '.$amazonIntegrationInstance->id.' '.AmazonReportTypeEnum::FBA_REPORT_INVENTORY_LEDGER_SUMMARY())
                        ->daily()
                        ->withoutOverlapping()
                        ->runInBackground()
                        ->name('Refresh FBA Inventory Ledger Summary for '.$amazonIntegrationInstance->name);

                    /*
                     * FBA Detail Reports
                     */

                    $frequency = config('app.env') == 'production' ? '0 1 * * *' : '0 3 * * *';

                    $events[] = $schedule
                        ->command('amazon:reports:request '.$amazonIntegrationInstance->id.' '.AmazonReportTypeEnum::FBA_REPORT_CUSTOMER_RETURNS())
                        ->cron($frequency)
                        ->withoutOverlapping()
                        ->runInBackground()
                        ->name('Refresh FBA Report (Customer Returns) '.$amazonIntegrationInstance->name);

                    $events[] = $schedule
                        ->command('amazon:reports:request '.$amazonIntegrationInstance->id.' '.AmazonReportTypeEnum::FBA_REPORT_SHIPMENTS())
                        ->cron($frequency)
                        ->withoutOverlapping()
                        ->runInBackground()
                        ->name('Refresh FBA Report (Shipments) '.$amazonIntegrationInstance->name);

                    $events[] = $schedule
                        ->command('amazon:reports:request '.$amazonIntegrationInstance->id.' '.AmazonReportTypeEnum::FBA_REPORT_REMOVAL_ORDERS())
                        ->cron($frequency)
                        ->withoutOverlapping()
                        ->runInBackground()
                        ->name('Refresh FBA Report (Removal Orders) '.$amazonIntegrationInstance->name);

                    $events[] = $schedule
                        ->command('amazon:reports:request '.$amazonIntegrationInstance->id.' '.AmazonReportTypeEnum::FBA_REPORT_REMOVAL_SHIPMENTS())
                        ->cron($frequency)
                        ->withoutOverlapping()
                        ->runInBackground()
                        ->name('Refresh FBA Report (Removal Shipments) '.$amazonIntegrationInstance->name);

                    $events[] = $schedule
                        ->command('amazon:reports:request '.$amazonIntegrationInstance->id.' '.AmazonReportTypeEnum::FBA_REPORT_RESTOCK())
                        ->daily()
                        ->withoutOverlapping()
                        ->runInBackground()
                        ->name('FBA Restock Report '.$amazonIntegrationInstance->name);

                    $events[] = $schedule
                        ->command('amazon:reports:sync '.$amazonIntegrationInstance->id)
                        ->everyMinute()
                        ->name('Sync Amazon Reports Statuses')
                        ->withoutOverlapping();

                    $events[] = $schedule
                        ->command('amazon:products:create-sku-products '.$amazonIntegrationInstance->id)
                        ->everyMinute()
                        ->name('Create SKU Products from Amazon')
                        ->withoutOverlapping();

                    $events[] = $schedule
                        ->command('amazon:orders:refresh '.$amazonIntegrationInstance->id)
                        ->everyMinute()
                        ->name('Refresh Amazon Orders')
                        ->withoutOverlapping();

//                    $events[] = $schedule
//                        ->command('amazon:inbound-shipments:refresh '.$amazonIntegrationInstance->id)
//                        ->everyFifteenMinutes()
//                        ->name('Refresh Amazon FBA Inbound Shipments')
//                        ->withoutOverlapping();

                    $events[] = $schedule
                        ->command('amazon:inbound-plans:refresh '.$amazonIntegrationInstance->id)
                        ->everyFifteenMinutes()
                        ->name('Refresh Amazon FBA Inbound Plans')
                        ->withoutOverlapping();

                    $events[] = $schedule
                        ->command('amazon:finances:groups:refresh '.$amazonIntegrationInstance->id)
                        ->hourly()
                        ->withoutOverlapping()
                        ->runInBackground()
                        ->name('Refresh Financial Event Groups');

                    $events[] = $schedule
                        ->command('amazon:financial-events:refresh '.$amazonIntegrationInstance->id)
                        ->hourly()
                        ->withoutOverlapping()
                        ->runInBackground()
                        ->name('Refresh Financial Events');

                    $events[] = $schedule
                        ->command('amazon:reports:refresh-settlement '.$amazonIntegrationInstance->id)
                        ->hourly()
                        ->withoutOverlapping()
                        ->runInBackground()
                        ->name('Refresh FBA Settlement Report');

                    $events[] = $schedule
                        ->command('amazon:outbound-fulfillments:refresh')
                        ->everyFifteenMinutes()
                        ->withoutOverlapping()
                        ->runInBackground()
                        ->name('Refresh Amazon Outbound Fulfillments');

                    if (@$amazonIntegrationInstance->integration_settings['automatically_create_warehouse_transfers_from_inbounds'] || @$amazonIntegrationInstance->integration_settings['automatically_create_purchase_orders_from_inbounds']) {
                        $events[] = $schedule
                            ->job(new AmazonProcessUnprocessedNewInboundShipmentsJob($amazonIntegrationInstance))
                            ->twiceDaily()
                            ->withoutOverlapping()
                            ->name('Process Unprocessed Amazon FBA Inbound Shipments');
                    }

                    if (@$amazonIntegrationInstance->integration_settings['automatically_create_warehouse_transfers_from_removal_orders']) {
                        $events[] = $schedule
                            ->job(new AmazonProcessUnprocessedRemovalOrdersJob($amazonIntegrationInstance))
                            ->twiceDaily()
                            ->withoutOverlapping()
                            ->name('Process Unprocessed Amazon FBA Removal Orders');
                    }
                }
            });

            $events[] = $schedule
                ->command('amazon:orders:reset-pii-data')
                ->daily()
                ->name('Reset Amazon PII Data')
                ->withoutOverlapping();
        }

        // Shopify
        IntegrationInstance::query()->shopify()->where('is_automatic_sync_enabled', true)->each(function (IntegrationInstance $integrationInstance) use (&$events, &$schedule) {
            $events[] = $schedule->command('sku:shopify:webhook-events:process '.$integrationInstance->id)
                ->everyMinute()
                ->withoutOverlapping();

            $schedule->job(new GenerateCreateSalesOrderFromShopifyOrderJobJobs($integrationInstance), 'salesOrderProcessing')
                ->withoutOverlapping()
                ->everyMinute();

            $schedule->job(new GenerateUpdateSalesOrderFromShopifyOrderJobJobs($integrationInstance), 'salesOrderProcessing')
                ->withoutOverlapping()
                ->everyMinute();

            $events[] = $schedule->command('sku:shopify:webhook-events:purge '.$integrationInstance->id)
                ->daily()
                ->withoutOverlapping();

            // TODO: Currently not working properly.  Should be refactored in SKU-6271
            //            $events[] = $schedule->command('sku:shopify:sync-missing-orders '.$integrationInstance->id)
            //                ->name(\App\Console\Commands\Shopify\GetOrdersCommand::class)
            //                ->withoutOverlapping()
            //                ->daily();
            $events[] = $schedule->command('sku:shopify:submit-fulfillments -i '.$integrationInstance->id)
                ->name(ShopifySubmitFulfillments::class)
                ->withoutOverlapping()
                ->daily();

            $events[] = $schedule->command('sku:shopify:auto-download-products '.$integrationInstance->id)
                ->name(\App\Jobs\Shopify\ShopifyGetProducts::class)
                ->withoutOverlapping()
                ->runInBackground()
                ->daily();

            $events[] = $schedule->command('sku:shopify:delete-missing-product-variants '.$integrationInstance->id)
                ->name(ShopifyDeleteMissingProductVariantsJob::class)
                ->withoutOverlapping()
                ->runInBackground()
                ->daily();
        });

        // WooCommerce
        WooCommerceIntegrationInstance::query()->where('is_automatic_sync_enabled', true)->each(function (WooCommerceIntegrationInstance $integrationInstance) use (&$events, &$schedule) {
            $schedule->job(new RefreshWooCommerceOrdersJob($integrationInstance, new WooCommerceGetOrdersAdt()), 'sales-channels')
                ->withoutOverlapping()
                ->everyFiveMinutes();

            $schedule->job(new CreateSkuOrdersFromWooCommerceJob($integrationInstance), 'salesOrderProcessing')
                ->withoutOverlapping()
                ->everyFiveMinutes();

            $schedule->job(new RefreshWooCommerceProductsJob($integrationInstance, new WooCommerceGetProductsAdt()), 'sales-channels')
                ->withoutOverlapping()
                ->daily();

            // TODO: Woocommerce submitting fulfillments
            //            $events[] = $schedule->command('sku:woocommerce:submit-fulfillments -i '.$integrationInstance->id)
            //                ->name(SubmitFulfillments::class)
            //                ->withoutOverlapping()
            //                ->daily();
        });

        // Ebay
        EbayIntegrationInstance::query()->where('is_automatic_sync_enabled', true)->each(function (EbayIntegrationInstance $integrationInstance) use (&$events, &$schedule) {
            $schedule->job(new RefreshEbayOrdersJob($integrationInstance, new EbayGetOrdersAdt()), 'sales-channels')
                ->withoutOverlapping()
                ->everyFiveMinutes();

            $schedule->job(new CreateSkuOrdersFromEbayOrdersJob($integrationInstance), 'salesOrderProcessing')
                ->withoutOverlapping()
                ->everyFiveMinutes();

            $schedule->job(new RefreshEbayLegacyProductsJob($integrationInstance, new EbayGetLegacyProductsAdt()), 'sales-channels')
                ->withoutOverlapping()
                ->daily();
        });

        /** @see SKU-4472 */
        // fix broken shopify orders and attempt to recreate them
        /* $events[] = $schedule->command('sku:shopify:fix-order-links -r')
                          ->name(FixShopifyToSkuOrderLink::class)
                          ->withoutOverlapping()
                          ->runInBackground()
                          ->everyFifteenMinutes();*/

        /** @see SKU-4921 */
        // fix shopify orders with errors and attempt to recreate them
        //        $events[] = $schedule->command('sku:shopify:process-orders')
        //      ->name(ProcessOrders::class)
        //      ->withoutOverlapping()
        //      ->runInBackground()
        //      ->everyFifteenMinutes();

        // Magento
        IntegrationInstance::query()->magento()->where('is_automatic_sync_enabled', true)->each(function (IntegrationInstance $integrationInstance) use (&$events, &$schedule) {
            $events[] = $schedule->command('sku:magento:auto-download-orders '.$integrationInstance->id)
                ->withoutOverlapping()
                ->everyFifteenMinutes();
            $events[] = $schedule->command(
                'sku:magento:auto-download-orders '.$integrationInstance->id,
                ['--lastUpdatedAfter' => now()->subHours(48), '--lastUpdatedBefore' => now()->subHours(3), '--downloadedBy' => DownloadedBy::JobBackup]
            )
                ->withoutOverlapping()
                ->daily();

            $events[] = $schedule->command('sku:magento:auto-download-products '.$integrationInstance->id)
                ->withoutOverlapping()
                ->everyFifteenMinutes();
        });

        /*
        |--------------------------------------------------------------------------
        | Shipping Providers
        |--------------------------------------------------------------------------
        */

        // ShipStation
        if (IntegrationInstance::query()->shipstation()->where('is_automatic_sync_enabled', true)->exists()) {
            $events[] = $schedule->command('sku:shipstation:submit-orders')->everyTenMinutes()->description(\App\Jobs\ShipStation\SubmitOrders::class);
            $events[] = $schedule->command('sku:shipstation:get-shipments --shipDateStart='.now()->format('Y-m-d'))->everyTenMinutes()->description(GetShipments::class);
        }

        // Starshipit
        if (IntegrationInstance::query()->starshipit()->where('is_automatic_sync_enabled', true)->exists()) {
            $events[] = $schedule->command('sku:starshipit:get-tracking')->daily()->description(\App\Jobs\Starshipit\GetTrackingJob::class);
        }

        // Veracore
        if (VeracoreIntegrationInstance::query()->where('is_automatic_sync_enabled', true)->exists()) {
            $events[] = $schedule->job(app(VeracoreUpdateTrackingJob::class))->everyTenMinutes()->description(VeracoreUpdateTrackingJob::class)->withoutOverlapping();
        }

        /* ShipMyOrders */
        if (ShipMyOrdersIntegrationInstance::query()->where('is_automatic_sync_enabled', true)->exists()) {
            $events[] = $schedule->job(app(UpdateSMOTrackingInfo::class))->everyTenMinutes()->description(UpdateSMOTrackingInfo::class)->withoutOverlapping();
            $events[] = $schedule->job(app(ShipMyOrdersRefreshInventoryJob::class))->everyThirtyMinutes()->description(ShipMyOrdersRefreshInventoryJob::class)->withoutOverlapping();
        }

        /*
        |--------------------------------------------------------------------------
        | Inventory Forecasting
        |--------------------------------------------------------------------------
        */

        // Inventory forecast job
        $events[] = $schedule->command('sku:inventory:forecast')
            ->dailyAt(
                Helpers::utcStartOfLocalDate(now())->format('H:i')
            )->description(ForecastInventoryCommand::class);

        /*
        |--------------------------------------------------------------------------
        | Purchasing & Backorders
        |--------------------------------------------------------------------------
        */

        app(SupplierRepository::class)->getSuppliersEligibleForBulk()->each(function (Supplier $supplier) use (&$schedule, &$events) {
            if ($supplier->po_batch_schedule) {
                foreach ($supplier->po_batch_schedule as $scheduleItem) {
                    foreach ($scheduleItem['days'] as $day) {
                        $events[] = $schedule->job(new PurchaseOrderSubmissionBatchingJob($supplier))
                            ->weeklyOn(Carbon::parse($day)->dayOfWeekIso, $scheduleItem['time'])
                            ->timezone('UTC');
                    }
                }
            }
        });

        app(SupplierRepository::class)->getSuppliersEligibleForBackorderPurchasing()->each(function (Supplier $supplier) use (&$schedule, &$events) {
            if ($supplier->backorder_po_schedule) {
                foreach ($supplier->backorder_po_schedule as $scheduleItem) {
                    foreach ($scheduleItem['days'] as $day) {
                        $events[] = $schedule->job(new BackorderPurchasingJob($supplier), 'backorderProcessing')
                            ->weeklyOn(Carbon::parse($day)->dayOfWeekIso, $scheduleItem['time'])
                            ->timezone('UTC');
                    }
                }
            }
        });

        // Attempt to release hanging backorder queues not received by PO receipts.
        $schedule->call(function () {
            if (Helpers::isJobRunning(ProcessNewInventory::class)) {
                return;
            }
            Artisan::call('sku:release-backorder-queues');
        })
            ->name(ReleaseBackorderQueues::class)
            ->everyFifteenMinutes()
            ->withoutOverlapping();

        $schedule->command('sku:refresh-has-backorder-cache')->daily();

        /*
        |--------------------------------------------------------------------------
        | Inventory
        |--------------------------------------------------------------------------
        */

        if (\App::environment() != 'testing') {
            /// Caches only for those listings needing updates based on product inventory updated_at > product_listings updated_at
            $this->integrationInstanceRepository->getSalesChannelIntegrationInstances()->each(function (IntegrationInstance $integrationInstance) use (&$schedule) {
                $schedule->job(new GenerateCacheProductListingQuantityJob($integrationInstance))->everyFiveMinutes();
            });
        }

        /*
        |--------------------------------------------------------------------------
        | Cleanup
        |--------------------------------------------------------------------------
        */

        // Purge model import data files
        $events[] = $schedule->command('sku:clean-import-exports')->daily()->description(RemoveImportExportFiles::class);

        $events[] = $schedule->command('telescope:prune --hours=24')->daily();
        $schedule->command('queue:prune-batches --hours=72 --unfinished=96')->daily(); // only the finished batches
        $schedule->command('queue:alert-batches')->hourly();
        $schedule->command('sku:delete:old-api-logs')->daily();
        $schedule->command('sku:delete:old-tracked-job-logs')->daily();
        $schedule->command('sku:prune-task-status')->daily();

        /*
        |--------------------------------------------------------------------------
        | System & Other
        |--------------------------------------------------------------------------
        */

        $schedule->command('horizon:snapshot')->everyFiveMinutes();
        $schedule->command('websockets:clean')->daily();

        // Delete files in storage/app/public/reports and subdirectories
        $schedule->command('sku:delete-old-report-files 7')->daily();

        // Inventory integrity script identify script
        // $schedule->command('sku:inventory:integrity:identify')->daily();

        if (config('innodb.monitor_locks_enabled')) {
            $schedule->command('sku:monitor-innodb-locks')->everyMinute();
        }
    }

    /**
     * Register the commands for the application.
     */
    protected function commands(): void
    {
        $this->load(__DIR__.'/Commands');

        require base_path('routes/console.php');
    }
}
