<?php

namespace App\Console\Commands;

use App\Models\InventoryMovement;
use Exception;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Throwable;

class UnifyInventoryMovementsCommand extends Command
{
    protected $signature = 'sku:inventory:unify-movements';

    protected $description = 'Unify duplicate inventory movements by summing their quantities and keeping the oldest record';

    /**
     * @throws Throwable
     */
    public function handle(): void
    {
        // Start the transaction
        DB::beginTransaction();

        try {
            // Find duplicate inventory movements and sum quantities
            $duplicates = InventoryMovement::select(
                'inventory_movement_date',
                'product_id',
                'link_id',
                'link_type',
                'type',
                'layer_id',
                'layer_type',
                'inventory_status',
                DB::raw('SUM(quantity) as total_quantity'),
                DB::raw('COUNT(*) as duplicate_count')
            )
                ->groupBy(
                    'inventory_movement_date',
                    'product_id',
                    'link_id',
                    'link_type',
                    'type',
                    'layer_id',
                    'layer_type',
                    'inventory_status'
                )
                ->having('duplicate_count', '>', 1)
                ->get();

            foreach ($duplicates as $duplicate) {
                // Get all duplicate movements, ordered by their creation date (oldest first)
                $movements = InventoryMovement::where('inventory_movement_date', $duplicate->inventory_movement_date)
                    ->where('product_id', $duplicate->product_id)
                    ->where('link_id', $duplicate->link_id)
                    ->where('link_type', $duplicate->link_type)
                    ->where('type', $duplicate->type)
                    ->where('layer_id', $duplicate->layer_id)
                    ->where('layer_type', $duplicate->layer_type)
                    ->where('inventory_status', $duplicate->inventory_status)
                    ->orderBy('created_at', 'asc')
                    ->get();

                // Keep the oldest record (first in the collection)
                $oldestMovement = $movements->first();

                // Calculate the total quantity (including the oldest record)
                $totalQuantity = $duplicate->total_quantity;

                // Update the quantity of the oldest movement
                $oldestMovement->update([
                    'quantity' => $totalQuantity
                ]);

                // Delete the other duplicate records, keeping the oldest one
                InventoryMovement::where('inventory_movement_date', $duplicate->inventory_movement_date)
                    ->where('product_id', $duplicate->product_id)
                    ->where('link_id', $duplicate->link_id)
                    ->where('link_type', $duplicate->link_type)
                    ->where('type', $duplicate->type)
                    ->where('layer_id', $duplicate->layer_id)
                    ->where('layer_type', $duplicate->layer_type)
                    ->where('inventory_status', $duplicate->inventory_status)
                    ->where('id', '!=', $oldestMovement->id)  // Exclude the oldest record
                    ->delete();
            }

            // Commit the transaction
            DB::commit();

            $this->info('Duplicate inventory movements unified successfully, keeping the oldest record.');
        } catch (Exception $e) {
            // Rollback the transaction in case of error
            DB::rollBack();
            $this->error('An error occurred: ' . $e->getMessage());
        }
    }
}
