<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\DB;
use Symfony\Component\Console\Command\Command as CommandAlias;
use Symfony\Component\Console\Input\InputOption;

class TruncateDatabase extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'sku:db:truncate {--seed} {--seeder=}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Truncates the tables in the database during testing.';

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        $this->truncateTables();
        if ($this->needsSeeding()) {
            $this->runSeeder();
        }

        return CommandAlias::SUCCESS;
    }

    public function truncateTables(): void
    {
        // This operation is only available for MySQL.
        if (DB::getDefaultConnection() !== 'mysql' || ! $this->testing()) {
            return;
        }

        // We get the tables in the database
        $queries = DB::select($this->getDatabaseTablesQuery());

        $queries = collect(array_values(array_map(function ($q) {
            return (array) $q;
        }, $queries)))->pluck('TABLE_NAME')->toArray();

        $query = rtrim(implode(' ', $queries), ';');

        $this->info('Truncating database');
        // We truncate all the tables in the database.
        DB::unprepared("SET FOREIGN_KEY_CHECKS=0; $query; SET FOREIGN_KEY_CHECKS=1;");
    }

    private function getDatabaseTablesQuery(): string
    {
        $database = config('database.connections.mysql.database');

        return "SELECT CONCAT('TRUNCATE TABLE ',table_schema,'.',TABLE_NAME, ';') AS TABLE_NAME FROM 
    INFORMATION_SCHEMA.TABLES WHERE table_schema = '$database'";
    }

    private function needsSeeding(): bool
    {
        return $this->option('seed') || $this->option('seeder');
    }

    private function runSeeder(): void
    {
        $this->call('db:seed', array_filter([
            '--database' => 'mysql',
            '--class' => $this->option('seeder') ?: 'Database\\Seeders\\DatabaseSeeder',
            '--force' => true,
        ]));
    }

    private function testing(): bool
    {
        return App::environment() === 'testing';
    }

    /**
     * @return array[]
     */
    protected function getOptions(): array
    {
        return [
            ['seed', null, InputOption::VALUE_NONE, 'Indicates if the seed task should be re-run'],
            ['seeder', null, InputOption::VALUE_OPTIONAL, 'The class name of the root seeder'],
        ];
    }
}
