<?php

namespace App\Console\Commands\Starshipit;

use App\Integrations\Starshipit;
use App\Jobs\Starshipit\GetTrackingJob;
use App\Models\IntegrationInstance;
use App\Models\SalesOrder;
use App\Models\Starshipit\StarshipitOrder;
use App\Services\SalesOrder\FulfillSalesOrderService;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;

class ImportFulfillments extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'sku:starshipit:import-fulfillments
                            {ids : Specify order ids }
                            {--s|sequence= : Specify sequence }';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Import fulfillments from StarShipIt';

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     */
    public function handle(): void
    {
        $ids = explode(',', $this->argument('ids'));
        $order_numbers = [];

        $starshipitIntegrationInstance = IntegrationInstance::with([])->starshipit()->firstOrFail();

        $starshipit = new Starshipit($starshipitIntegrationInstance);

        $sequence = $this->option('sequence') ? $this->option('sequence') : 1;

        $this->info('Using sequence '.$sequence);

        foreach ($ids as $id) {
            $salesOrder = SalesOrder::with([])->findOrFail($id);
            echo $salesOrder->sales_order_number."\n";
            if ($salesOrder->fulfillment_status == SalesOrder::FULFILLMENT_STATUS_FULFILLED) {
                Log::info("Skipping $salesOrder->sales_order_number since not fulfilled");

                continue;
            }
            $getOrderResponse = $starshipit->getOrder($salesOrder->sales_order_number.'.'.$sequence, 'order_number');

            if ($getOrderResponse->statusCode == 200 && $getOrderResponse->body['success']) {
                /*
                 * Now we can use this payload to create the sales order fulfillment linked to starshipit
                 */

                $ssiOrder = $getOrderResponse->body['order'];

                /** @var \App\Models\Starshipit\Mysql\Order $starshipitOrder */
                $starshipitOrder = null;

                $starshipitOrder = StarshipitOrder::with([])->firstOrNew(['order_id' => $ssiOrder['order_id']]);

                $ssiLines = [];

                foreach ($ssiOrder['items'] as $item) {
                    $matchingLine = $salesOrder->getMatchingUnfulfilledSalesOrderLineFromSku($item['sku']);
                    $warehouse_id = $matchingLine['warehouse_id'];
                    $ssiLines[] = [
                        'quantity' => $item['quantity'],
                        'sales_order_line_id' => $matchingLine['id'],
                    ];
                }

                $request = [
                    'fulfilled_at' => $ssiOrder['order_date'],
                    'fulfillment_lines' => $ssiLines,
                    'fulfillment_type' => 'starshipit',
                    'fulfillment_sequence' => $sequence,
                    'metadata' => json_encode(['signature_required' => $ssiOrder['signature_required']]),
                    'requested_shipping_method' => $ssiOrder['shipping_method'] ?? '',
                    'warehouse_id' => $warehouse_id,
                ];

                $fulfillment = FulfillSalesOrderService::make($salesOrder)->fulfillWithInputs($request, true, false);

                if (! $starshipitOrder->exists) {
                    try {
                        $starshipitOrder->json_object = $ssiOrder;
                        $starshipitOrder->errors = null;

                        $starshipitOrder->sku_fulfillment_id = $fulfillment->id;
                        $starshipitOrder->save();
                    } catch (\Throwable $exception) {
                        Log::debug("CreateSkuFulfillmentFromStarShipIt, order_id:{$ssiOrder['order_id']}: {$exception->getMessage()}", $exception->getTrace());
                        try {
                            $starshipitOrder->rawOrder = $ssiOrder;
                            $starshipitOrder->errors = $exception->getMessage();
                            $starshipitOrder->save();
                        } catch (\Throwable $exception) {
                            Log::debug("CreateSkuFulfillmentFromStarShipIt(saving raw), order_id:{$ssiOrder['order_id']}: {$exception->getMessage()}", $exception->getTrace());
                        }
                    }
                } else {
                    Log::info('SSI order '.$ssiOrder['order_id'].' already exists, but creating sku fulfillments');
                    $starshipitOrder->sku_fulfillment_id = $fulfillment->id;
                    $starshipitOrder->save();
                }

                foreach ($ssiOrder['packages'] as $shippedOrder) {
                    $trackingInfo = [
                        'status' => 'Manifested', // at least the status is "Manifested" because we are fetched the Shipped orders
                        'carrier_name' => $ssiOrder['carrier_name'],
                        'carrier_service' => $ssiOrder['carrier_service_code'], // "code" because the webhook and tracking API return the "code" only
                        'tracking_number' => $shippedOrder['tracking_number'],
                        'shipment_date' => $shippedOrder['label_created_date'],
                        'tracking_status' => $shippedOrder['delivery_status'] ?? 'Delivered',
                        'received_by' => 'RequestTrackingUpdatesService',
                        'price_breakdown' => $ssiOrder['price_breakdown'] ?? null,
                    ];

                    // update sales order fulfillment
                    GetTrackingJob::addTrackingToSalesOrderFulfillment($fulfillment, $trackingInfo);
                }

                Log::info("Imported fulfillment for $starshipitOrder->sales_order_number (".$ssiOrder['order_id'].')', $request);
            }
        }
    }
}
