<?php

namespace App\Console\Commands\Shopify;

use App\Models\SalesOrderFulfillment;
use App\Services\SalesOrderFulfillment\SubmitTrackingInfo;
use Illuminate\Console\Command;
use PHPShopify\Exception\CurlException;

class ShopifySubmitFulfillments extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'sku:shopify:submit-fulfillments 
                                {--i|integrationInstance= : The ID of the integration instance}
                                {--o|order_id= : Specific order id }
                                {--f|sales_order_fulfillment_id= : Specific sales order fulfillment id}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Submits fulfillments not yet submitted to Shopify';

    /**
     * Execute the console command.
     */
    public function handle(): void
    {
        $integration_instance_id = (int) $this->option('integrationInstance') ?? null;

        $query = SalesOrderFulfillment::with([])
            ->whereNull('submitted_to_sales_channel_at')
            ->where('status', SalesOrderFulfillment::STATUS_FULFILLED)
            ->whereHas('salesOrder', function ($builder) use ($integration_instance_id) {
                return $builder->when(! is_null($this->option('order_id')), function ($builder) {
                    return $builder->where('id', $this->option('order_id'));
                })->whereHas('salesChannel.integrationInstance', function ($builder) use ($integration_instance_id) {
                    return $integration_instance_id ? $builder->where('id', $integration_instance_id) : $builder;
                })->whereNull('canceled_at');
            });

        if ($this->option('sales_order_fulfillment_id')) {
            $query->where('id', $this->option('sales_order_fulfillment_id'));
        }

        $query->orderBy('created_at', 'desc');

        $affected = $query->count();

        $this->line("{$affected} SKU order(s) to submit.");

        $query->each(function (SalesOrderFulfillment $fulfillment) {
            $this->output->text("Submitting fulfillment and tracking for: {$fulfillment->salesOrder->sales_order_number}.\n");
            try {
                $response = SubmitTrackingInfo::factory($fulfillment)?->submit();
            } catch (CurlException) {
                $this->warn("{$fulfillment->salesOrder->sales_order_number} Failed due to timeout\n");
                return;
            }

            $this->line($response['message'] ?? 'Nothing to submit');
            if (! $response || ! $response['success']) {
                $this->warn("{$fulfillment->salesOrder->sales_order_number} Failed.\n");
            } else {
                $this->info("{$fulfillment->salesOrder->sales_order_number} Submitted.\n");
            }
        });

        $this->line("{$affected} SKU order(s) submitted.");
    }
}
