<?php

namespace App\Console\Commands\Shopify;

use App\Helpers;
use App\Models\Setting;
use App\Models\Shopify\ShopifyOrder;
use App\Services\StockTake\OpenStockTakeException;
use Carbon\Carbon;
use Illuminate\Console\Command;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Log;

class ShopifyProcessOrders extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'sku:shopify:process-orders 
                                {--i|integrationInstance= : The ID of the integration instance}
                                {--o|open : Open orders only }
                                {--c|closed : Closed orders only}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Creates sku orders from sales channel orders';

    /**
     * Execute the console command.
     */
    public function handle(): void
    {
        //Log::info('Running shopify process orders');

        $integration_instance_id = (int) $this->option('integrationInstance') ?? null;
        $open = $this->option('open') ?? false;
        $closed = $this->option('closed') ?? false;

        $query = \App\Models\Shopify\ShopifyOrder::query();

        $query->with(['integrationInstance'])
            ->whereHas('integrationInstance')
            ->whereNull('sku_sales_order_id')
            ->whereNull('archived_at');

        if ($open) {
            $query->where(function ($query) {
                $query->whereNull('fulfillment_status');
                $query->orWhere('fulfillment_status', '=', 'partial');
            });
        }
        if ($closed) {
            $query->whereIn('fulfillment_status', ['fulfilled', 'restocked']);
        }

        if ($integration_instance_id) {
            $query->where('integration_instance_id', $integration_instance_id);
        }

        // after inventory start date
        $inventoryStartDate = Helpers::setting(Setting::KEY_INVENTORY_START_DATE);
        if ($inventoryStartDate) {
            $this->whereGte($query, 'created_at', Carbon::parse($inventoryStartDate, 'UTC')->toIso8601String());
        }

        $affected = $query->count();

        $this->output->text("{$affected} SKU order(s) to be created.");

        $query->each(function (ShopifyOrder $order) {
            if ($order->shouldSkip()) {
                $this->output->text("order {$order->name} skipped.\n");

                return; // continue
            }

            $this->output->text("Creating SKU order for: {$order->name}.\n");
            try {
                Helpers::handleDuplicateAndDeadlockSQLExceptions(fn () => $order->createSKUOrder());
                $this->output->text("SKU order created for: {$order->order_number}.\n");
            } catch (OpenStockTakeException $openStockTakeException) {
                $this->output->text("failed to create SKU order for: {$order->order_number}, {$openStockTakeException->getMessage()}.\n");
            }
        });

        $this->output->text("{$affected} SKU order(s) created.");
        //Log::info("{$affected} SKU order(s) created.");
    }

    /**
     * filter where date greater than or equal.
     */
    private function whereGte(Builder $builder, string $attribute, string $value): Builder
    {
        return $builder->where($attribute, '>=', $value);
    }
}
