<?php

namespace App\Console\Commands\Shopify;

use App\Enums\DownloadedBy;
use App\Models\IntegrationInstance;
use Carbon\Carbon;
use Illuminate\Console\Command;

class ShopifyGetProducts extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'sku:shopify:get-products 
                          {integrationInstance : The ID of the integration instance}
                          {--createdAfter= : Return products created after a specified date. }
                          {--createdBefore= : Return products created before a specified date. }
                          {--lastUpdatedAfter= : Return products last updated after a specified date. }
                          {--lastUpdatedBefore= : Return products last updated before a specified date. }
                          {--l|limit=250 : Return up to this many results per page. }';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Get Shopify products for the given integration instance.';

    /**
     * Execute the console command.
     */
    public function handle(): void
    {
        $integrationInstance = IntegrationInstance::with('integration')
            ->findOrFail($this->argument('integrationInstance'));
        if (! $integrationInstance || ! $integrationInstance->isShopify()) {
            $this->error('The integration must be Shopify.');

            return;
        }

        dispatch(new \App\Jobs\Shopify\ShopifyGetProducts($integrationInstance, $this->getCommandOptions()))->onQueue('sales-channels');
        //dispatch_now(new \App\Jobs\Shopify\GetProducts($integrationInstance, $this->getCommandOptions()));

        $this->info('Added Shopify products request to the Queue, it will be processed shortly.');
    }

    /**
     * Parse command options to Shopify options.
     */
    protected function getCommandOptions(): array
    {
        $options = ['downloaded_by' => DownloadedBy::Command];
        if ($this->option('createdAfter')) {
            $options['created_at_min'] = Carbon::parse($this->option('createdAfter'))->toISOString();
        }
        if ($this->option('createdBefore')) {
            $options['created_at_max'] = Carbon::parse($this->option('createdBefore'))->toISOString();
        }
        if ($this->option('lastUpdatedAfter')) {
            $options['updated_at_min'] = Carbon::parse($this->option('lastUpdatedAfter'))->toISOString();
        }
        if ($this->option('lastUpdatedBefore')) {
            $options['updated_at_max'] = Carbon::parse($this->option('lastUpdatedBefore'))->toISOString();
        }

        if ($this->option('limit')) {
            $options['limit'] = (int) $this->option('limit');
        }

        return $options;
    }
}
