<?php

namespace App\Console\Commands\Shopify;

use App\Integrations\Shopify;
use App\Models\IntegrationInstance;
use App\Models\Shopify\ShopifyProduct;
use App\Models\Shopify\ShopifyWebhook;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Throwable;

class ShopifyCreateWebhooksCommand extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'sku:shopify:create-webhooks
                                 {integrationInstance : The ID of the integration instance}
                                 {--force : trying to add webhooks to Shopify without checking if they are exist on DB}
                                 {--sync-from-shopify : Add webhook records from Shopify}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Create shopify webhooks to listen on inventoryItemDelete/inventoryLevelUpdate';

    /**
     * Execute the console command.
     */
    public function handle(): void
    {
        /** @var IntegrationInstance $integrationInstance */
        $integrationInstance = IntegrationInstance::with('integration')->findOrFail($this->argument('integrationInstance'));
        if (! $integrationInstance || ! $integrationInstance->isShopify()) {
            $this->error('The integration must be Shopify.');

            return;
        }

        $route = route('webhooks.shopify.inventory_item_delete', ['integration_instance' => $integrationInstance->id, 'token' => config('webhooks.shopify')]);

        $disallowed_addresses = [
            '127.0.0.1',
            'localhost',
        ];

        if (Str::contains($route, $disallowed_addresses) === false) {
            $this->createWebhook($integrationInstance, ShopifyWebhook::TOPIC_INVENTORY_ITEMS_DELETE, 'webhooks.shopify.inventory_item_delete');
            $this->createWebhook($integrationInstance, ShopifyWebhook::TOPIC_INVENTORY_ITEMS_UPDATE, 'webhooks.shopify.inventory_level_update');
            $this->createWebhook($integrationInstance, ShopifyWebhook::TOPIC_PRODUCTS_CREATE, 'webhooks.shopify.product_create');
            $this->createWebhook($integrationInstance, ShopifyWebhook::TOPIC_ORDERS_CREATE, 'webhooks.shopify.order_create');
            $this->createWebhook($integrationInstance, ShopifyWebhook::TOPIC_ORDERS_UPDATED, 'webhooks.shopify.order_updated');
            $this->createWebhook($integrationInstance, ShopifyWebhook::TOPIC_ORDER_TRANSACTIONS_CREATE, 'webhooks.shopify.order_transaction_create');

            /*
             * TODO: create webhook in graphql
             */
            $this->createWebhook($integrationInstance, 'bulk_operations/finish', 'webhooks.shopify.bulk_operations_finish');

            /*
             * Kalvin:
             * Until Shopify updates product webhooks/updated_at to not update for inventory updates, we are disabling
             * this webhook and its handling
             * See ref: https://community.shopify.com/c/shopify-apis-and-sdks/will-products-update-webhook-always-fire-when-an-inventory-level/td-p/572280
            /*
            $this->createWebhook($integrationInstance, 'products/update', 'webhooks.shopify.product_update');
            */

            if ($this->option('sync-from-shopify')) {
                $this->syncWebhooksFromShopify($integrationInstance);
            }

            // add sales channel quantities
            if (ShopifyWebhook::hasInventoryLevelUpdateWebhook($integrationInstance->id, true)) {
                ShopifyProduct::query()
                    ->where('integration_instance_id', $integrationInstance->id)
                    ->join('product_listings', 'product_listings.id', '=', 'shopify_products.product')
                    ->whereNull('product_listings.sales_channel_qty_last_updated')
                    ->update([
                        'product_listings.sales_channel_qty' => DB::raw('IFNULL(`product_listings`.`sales_channel_qty`, `shopify_products`.`inventory_quantity`)'),
                        'product_listings.sales_channel_qty_last_updated' => DB::raw('`shopify_products`.`updated_at`')]);
            }
        }
    }

    private function convertShopifyDateToUTC(string $jsonColumn, string $jsonProperty): string
    {
        $plusOrMinus = "IF(LOCATE('+', JSON_UNQUOTE(JSON_EXTRACT(`$jsonColumn`, '$.$jsonProperty'))), '+', '-')";
        $datetimeWithoutTimezone = "SUBSTRING_INDEX(JSON_UNQUOTE(JSON_EXTRACT(`$jsonColumn`, '$.$jsonProperty')), $plusOrMinus, 1)";
        $strToDate = "STR_TO_DATE($datetimeWithoutTimezone, '%Y-%m-%dT%H:%i:%s')";
        $timezoneHours = "SUBSTRING_INDEX(JSON_UNQUOTE(JSON_EXTRACT(`$jsonColumn`, '$.$jsonProperty')), $plusOrMinus, -1)";

        return "CONVERT_TZ($strToDate, CONCAT($plusOrMinus, $timezoneHours), 'UTC')";
    }

    private function createWebhook(IntegrationInstance $integrationInstance, string $topic, string $route_name, string $api = 'rest')
    {
        $webhook = ShopifyWebhook::getWebhooks($integrationInstance->id)->firstWhere('topic', $topic);
        if ($webhook && ! $this->option('force')) {
            return;
        }

        $shopify = new Shopify($integrationInstance);
        try {
            $this->info("Creating webhook for $topic");
            // create the webhook on Shopify
            $shopifyWebhook = $shopify->createWebhook([
                'topic' => $topic,
                'address' => route($route_name, ['integration_instance' => $integrationInstance->id, 'token' => config('webhooks.shopify')]),
                'format' => 'json',
            ], $api);
            // save it on the database
            ShopifyWebhook::query()->upsert(['integration_instance_id' => $integrationInstance->id, 'json_data' => json_encode($shopifyWebhook)], []);
        } catch (Throwable $exception) {
            $this->error("$topic: ".$exception->getMessage());
        }
    }

    private function syncWebhooksFromShopify(IntegrationInstance $integrationInstance)
    {
        // remove webhook records of the integration instance
        ShopifyWebhook::query()->where('integration_instance_id', $integrationInstance->id)->delete();
        // get webhooks from Shopify and add them to shopify webhooks table
        $shopify = new Shopify($integrationInstance);
        foreach ($shopify->getWebhooks() as $webhook) {
            ShopifyWebhook::query()->upsert(['integration_instance_id' => $integrationInstance->id, 'json_data' => json_encode($webhook)], []);
        }
    }
}
