<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Storage;

/**
 * Class RemoveImportExportFiles.
 */
class RemoveImportExportFiles extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'sku:clean-import-exports';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Removes data import/export files that are more than 1 week old.';

    protected $excludedFiles = ['.gitignore'];

    /**
     * Execute the console command.
     */
    public function handle(): void
    {
        $this->purgeImports();
        $this->purgeExports();
    }

    private function purgeImports()
    {
        $this->purge('model-imports', 'import');
    }

    private function purgeExports()
    {
        $this->purge('model-exports', 'export');
    }

    private function purge(string $diskName, string $type)
    {
        $disk = Storage::disk($diskName);

        $files = $disk->files();
        $deletedCount = 0;

        foreach ($files as $file) {
            if (now()->subWeek()->timestamp > $disk->lastModified($file) && ! $this->fileExcluded($file)) {
                $disk->delete($file);
                $deletedCount++;
            }
        }

        $this->info("Purged $deletedCount $type files.");
    }

    private function fileExcluded(string $file): bool
    {
        return in_array($file, $this->excludedFiles);
    }
}
