<?php

namespace App\Console\Commands\Products;

use App\Exceptions\InsufficientStockException;
use App\Jobs\GenerateCacheProductListingQuantityJob;
use App\Jobs\ForecastInventoryJob;
use App\Jobs\InventorySnapshotJob;
use App\Jobs\RecacheProfitReportingJob;
use App\Jobs\UpdateProductsInventoryAndAvgCost;
use App\Models\FifoLayer;
use App\Models\InventoryAdjustment;
use App\Models\InventoryAssemblyLine;
use App\Models\InventoryForecastItem;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\ProductBlemished;
use App\Models\ProductComponent;
use App\Models\ProductListing;
use App\Models\PurchaseOrderLine;
use App\Models\SalesCreditLine;
use App\Models\SalesOrderFulfillmentLine;
use App\Models\SalesOrderLine;
use App\Models\SalesOrderLineLayer;
use App\Models\StockTakeItem;
use App\Models\WarehouseTransferLine;
use App\Services\InventoryManagement\InventoryManager;
use Illuminate\Console\Command;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\DB;

class DeprecateOldSkus extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'sku:products:patch:deprecate-old-skus
                                {pairs : comma separated ids of products, original-destination}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Merge one product into another';

    protected array $integrationInstances = [];

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        // TODO: Need to fix FixFifoLayerQuantityCacheJob first
        return 0;

        $pairs = explode(',', $this->argument('pairs'));

        $destinationProductIds = [];

        DB::beginTransaction();
        try {
            foreach ($pairs as $pair) {
                $productIds = explode('-', $pair);
                if (count($productIds) != 2 || $productIds[0] == $productIds[1]) {
                    throw new \Exception("invalid pair: {$pair}");
                }

                /** @var Product $originalProduct */
                $originalProduct = Product::with([])->findOrFail($productIds[0]);
                /** @var Product $destinationProduct */
                $destinationProduct = Product::with([])->findOrFail($productIds[1]);
                $this->info("Merging $originalProduct->sku ($originalProduct->id) into $destinationProduct->sku ($destinationProduct->id)");

                $destinationProductIds[] = $destinationProduct->id;

                // it should be first to handle fifo layers that should be deleted
                $this->moveInventoryRecords($originalProduct, $destinationProduct);

                $this->moveSalesRecords($originalProduct, $destinationProduct);

                $this->movePurchaseRecords($originalProduct, $destinationProduct);

                // We are trying to avoid creating stock take items without inventory movements.  This may be causing it.
                //$this->moveStockTakeRecords($originalProduct, $destinationProduct);

                $this->moveWarehouseTransferRecords($originalProduct, $destinationProduct);

                $this->moveBlemishedRecords($originalProduct, $destinationProduct);

                $this->moveBundleRecords($originalProduct, $destinationProduct);

                $this->moveProductListingRecords($originalProduct, $destinationProduct);

                $this->info('Deleting original product...');
                $originalProduct->refresh();
                $originalProduct->delete();
            }

            $this->info('Caching ProductsInventoryAndAvgCost...');
            (new UpdateProductsInventoryAndAvgCost($destinationProductIds))->handle();
            $this->info('Caching ProductsBenchmarks...');
            (new InventorySnapshotJob(productIds: $destinationProductIds))->handle();
            $this->info('Caching FifoLayerQuantityCache...');
            // run FixFifoLayerQuantityCache as a job
            // TODO: Fix this
            //(new FixFifoLayerQuantityCacheJob($destinationProductIds))->handle();
            $this->info('Caching ProfitReporting...');
            // recalculate for reporting products and orders
            (new RecacheProfitReportingJob($destinationProductIds))->handle();
            $this->info('Caching ForecastInventory...');
            // recache inventory forecasts
            /** @var ForecastInventoryJob $forecaster */
            $forecaster = app()->make(ForecastInventoryJob::class);
            $forecaster->withProductIds($destinationProductIds)->handle();

            /*
             * Loop through each integration instance belonging to the destination product listings to update caches
             */
            foreach (array_unique($this->integrationInstances) as $integrationInstance) {
                (new GenerateCacheProductListingQuantityJob($integrationInstance))->handle();
            }

            DB::commit();
        } catch (\Throwable $e) {
            DB::rollBack();
            dump($e->getMessage(), $e->getFile(), $e->getLine(), $e->getTraceAsString());
            throw $e;
        }

        return 0;
    }

    private function moveSalesRecords(Product $originalProduct, Product $destinationProduct)
    {
        $this->info('Sales order lines');
        $this->info($originalProduct->salesOrderLines()->count().' sales order lines');

        $originalProduct->salesOrderLines()->eachById(function (SalesOrderLine $salesOrderLine) use ($destinationProduct) {
            $this->info($salesOrderLine->salesOrder->sales_order_number." @ $salesOrderLine->quantity (ID: $salesOrderLine->id)");
            $salesOrderLine->product_id = $destinationProduct->id;
            $salesOrderLine->saveQuietly();
            /*
             * Will changing sales order line product update movements automatically? no, it will not
             * Will it update reporting orders automatically? yes, so I used saveQuietly to prevent do that
             */
        });

        $this->info('Sales Credit Lines');
        $this->info($originalProduct->salesCreditLines()->count().' sales credit lines');

        $originalProduct->salesCreditLines()->eachById(function (SalesCreditLine $salesCreditLine) use ($destinationProduct) {
            $this->info($salesCreditLine->salesCredit->sales_credit_number." @ $salesCreditLine->quantity (ID: $salesCreditLine->id)");
            $salesCreditLine->product_id = $destinationProduct->id;
            $salesCreditLine->save();
        });
    }

    private function movePurchaseRecords(Product $originalProduct, Product $destinationProduct)
    {
        $this->info('Purchase Order Lines');
        $this->info($originalProduct->purchaseOrderLines()->count().' purchase order lines');

        $originalProduct->purchaseOrderLines()->eachById(function (PurchaseOrderLine $purchaseOrderLine) use ($destinationProduct) {
            $this->info($purchaseOrderLine->purchaseOrder->purchase_order_number." @ $purchaseOrderLine->quantity (ID: $purchaseOrderLine->id)");
            $purchaseOrderLine->product_id = $destinationProduct->id;
            $purchaseOrderLine->save();
        });
    }

    private function moveInventoryRecords(Product $originalProduct, Product $destinationProduct)
    {
        $this->info('Inventory movements');
        $this->info($originalProduct->inventoryMovements()->count().' inventory movements');

        // handle/move the movements that should not be deleted
        $movementTypesToDelete = [InventoryMovement::TYPE_STOCK_TAKE];
        $fifoLayerTypesToDelete = collect(array_flip(FifoLayer::REQUEST_LINK_TYPES))->only($movementTypesToDelete)->values();

        $originalProduct
            ->inventoryMovements()
            ->where(function (Builder $builder) use ($movementTypesToDelete) {
                $builder->whereNotIn('type', $movementTypesToDelete);
                // We are trying to avoid creating stock take items without inventory movements.  This may be causing it.
                //              $builder->orWhere(function (Builder $builder) {
                //                  $builder->where('type', InventoryMovement::TYPE_STOCK_TAKE)
                //                          ->where('quantity', '<', 0);
                //              });
            })
            ->where(function (Builder $builder) use ($fifoLayerTypesToDelete) {
                $builder->where('layer_type', '!=', FifoLayer::class)
                    ->orWhereDoesntHaveMorph('layer', FifoLayer::class, fn ($q) => $q->whereIn('link_type', $fifoLayerTypesToDelete));
            })
            ->eachById(function (InventoryMovement $inventoryMovement) use ($destinationProduct) {
                $this->info("$inventoryMovement->type ($inventoryMovement->inventory_status) @ $inventoryMovement->quantity (ID: $inventoryMovement->id)");
                switch ($inventoryMovement->type) {
                    case 'sale':
                    case 'purchase_receipt':
                    case 'adjustment':
                    case 'transfer':
                    case 'return':
                    case 'assembly':
                    case 'stock_take':
                    case 'reclassification':
                        if ($inventoryMovement->fifo_layer) {
                            $inventoryMovement->fifo_layer->product_id = $destinationProduct->id;
                            $inventoryMovement->fifo_layer->save();
                        }

                        $inventoryMovement->product_id = $destinationProduct->id;
                        $inventoryMovement->save();
                        break;
                }
            });

        // handle/delete inventory movements for fifo layers that should be deleted
        $this->handleInventoryForDeletedFifoLayers($originalProduct, $destinationProduct);

        if ($count = $originalProduct->inventoryMovements()->count()) {
            throw new \Exception("There are still {$count} unhandled movements");
        }

        $this->info('Inventory Adjustments');
        $this->info($originalProduct->inventoryAdjustments()->count().' inventory adjustments');

        $originalProduct->inventoryAdjustments()->eachById(function (InventoryAdjustment $inventoryAdjustment) use ($destinationProduct) {
            $this->info("$inventoryAdjustment->quantity (ID: $inventoryAdjustment->id)");
            $inventoryAdjustment->product_id = $destinationProduct->id;
            $inventoryAdjustment->save();
        });

        $this->info('Inventory Assembly Lines');
        $this->info($originalProduct->inventoryAssemblyLines()->count().' inventory assembly lines');

        $originalProduct->inventoryAssemblyLines()->eachById(function (InventoryAssemblyLine $inventoryAssemblyLine) use ($destinationProduct) {
            $this->info("$inventoryAssemblyLine->product_type for ".$inventoryAssemblyLine->inventoryAssembly->action." @ $inventoryAssemblyLine->quantity (ID: $inventoryAssemblyLine->id)");
            $inventoryAssemblyLine->product_id = $destinationProduct->id;
            $inventoryAssemblyLine->save();
        });

        $this->info('Inventory Forecasts Cache');
        $this->info(InventoryForecastItem::query()->where('product_id', $originalProduct->id)->count().' inventory forecasts cache');

        InventoryForecastItem::query()->where('product_id', $originalProduct->id)->eachById(fn ($ifc) => $ifc->delete());
    }

    public function handleInventoryForDeletedFifoLayers(Product $originalProduct, Product $destinationProduct)
    {
        // get fifo layers that will delete and stock_take
        $typesToDelete = [FifoLayer::REQUEST_LINK_TYPE_ST];
        $fifoLayerTypesToDelete = collect(array_flip(FifoLayer::REQUEST_LINK_TYPES))->only($typesToDelete)->values();
        $fifoLayersToDelete = $originalProduct->fifoLayers()->whereIn('link_type', $fifoLayerTypesToDelete)->get();

        // get Fulfilled sales order lines that used these fifo layers by sales_order_line_layers table
        SalesOrderLine::with(['layers.layer', 'salesOrder', 'salesOrderFulfillmentLines'])
            ->where('product_id', $originalProduct->id)
            ->whereHas('layers', function (Builder $builder) use ($fifoLayersToDelete) {
                $builder->where('layer_type', FifoLayer::class)
                    ->whereIn('layer_id', $fifoLayersToDelete->pluck('id'));
            })
            ->whereHas('salesOrderFulfillmentLines')
            ->eachById(function (SalesOrderLine $salesOrderLine) use ($destinationProduct, $fifoLayersToDelete) {
                // change product_id to the destination product id
                $this->info($salesOrderLine->salesOrder->sales_order_number." @ $salesOrderLine->quantity (ID: $salesOrderLine->id)");
                $salesOrderLine->product_id = $destinationProduct->id;
                $salesOrderLine->saveQuietly();

                // remove movements of these layers to be able to add fulfillment movements
                $salesOrderLine->inventoryMovements()->where('layer_type', FifoLayer::class)->whereIn('layer_id', $fifoLayersToDelete->pluck('id'))->delete();

                // use InventoryManager::takeFromStock to reserve the quantity that was reserved by these fifo layers
                $layers = $salesOrderLine->layers->where('layer_type', FifoLayer::class)->whereIn('layer_id', $fifoLayersToDelete->pluck('id'));
                /** @var SalesOrderLineLayer $salesOrderLineLayer */
                foreach ($layers as $salesOrderLineLayer) {
                    try {
                        InventoryManager::with($salesOrderLine->warehouse_id, $destinationProduct)
                            ->takeFromStock($salesOrderLineLayer->quantity, $salesOrderLine);
                    } catch (InsufficientStockException $insufficientStockException) {
                        $this->handleInsufficientStockException($salesOrderLine, $salesOrderLineLayer->quantity, $salesOrderLineLayer->layer->avg_cost);
                        // take from stock again
                        InventoryManager::with($salesOrderLine->warehouse_id, $destinationProduct)
                            ->takeFromStock($salesOrderLineLayer->quantity, $salesOrderLine);
                    }
                }

                // use SalesOrderFulfillmentLine::reverseReservedInventoryMovements for fulfillment lines
                $salesOrderLine->salesOrderFulfillmentLines->each(function (SalesOrderFulfillmentLine $fulfillmentLine) {
                    $fulfillmentLine->inventoryMovements()->delete();
                    $fulfillmentLine->negateReservationMovements();
                });
            });

        // get Unfulfilled sales order lines that used these fifo layers by sales_order_line_layers table
        SalesOrderLine::with(['layers.layer', 'salesOrder'])
            ->where('product_id', $originalProduct->id)
            ->whereHas('layers', function (Builder $builder) use ($fifoLayersToDelete) {
                $builder->where('layer_type', FifoLayer::class)
                    ->whereIn('layer_id', $fifoLayersToDelete->pluck('id'));
            })
            ->eachById(function (SalesOrderLine $salesOrderLine) use ($destinationProduct, $fifoLayersToDelete) {
                // change product_id to the destination product id
                $this->info($salesOrderLine->salesOrder->sales_order_number." @ $salesOrderLine->quantity (ID: $salesOrderLine->id)");
                $salesOrderLine->product_id = $destinationProduct->id;
                $salesOrderLine->saveQuietly();

                // use InventoryManager::takeFromStock to reserve the quantity that was reserved by these fifo layers
                $layers = $salesOrderLine->layers->where('layer_type', FifoLayer::class)->whereIn('layer_id', $fifoLayersToDelete->pluck('id'));
                foreach ($layers as $salesOrderLineLayer) {
                    InventoryManager::with($salesOrderLine->warehouse_id, $destinationProduct)
                        ->takeFromStock($salesOrderLineLayer->quantity, $salesOrderLine, false, $salesOrderLine);
                }
            });

        // delete the fifo layer and its movements
        $fifoLayersToDelete->each(function (FifoLayer $fifoLayer) use ($destinationProduct) {
            $fifoLayer->inventoryMovements()->eachById(function (InventoryMovement $inventoryMovement) use (&$originalMovement, $destinationProduct) {
                // other types that have a negative quantity
                if ($inventoryMovement->type != InventoryMovement::TYPE_SALE && $inventoryMovement->quantity < 0) {
                    // change product_id to the destination product id
                    $inventoryMovement->link->product_id = $destinationProduct->id;
                    $inventoryMovement->link->save();

                    try {
                        InventoryManager::with($inventoryMovement->warehouse_id, $destinationProduct)
                            ->takeFromStock(abs($inventoryMovement->quantity), $inventoryMovement->link);
                    } catch (InsufficientStockException $insufficientStockException) {
                        $this->handleInsufficientStockException($inventoryMovement->link, abs($inventoryMovement->quantity));
                        // take from stock again
                        InventoryManager::with($inventoryMovement->warehouse_id, $destinationProduct)
                            ->takeFromStock(abs($inventoryMovement->quantity), $inventoryMovement->link);
                    }
                }

                $inventoryMovement->delete();
            });

            $fifoLayer->delete();
        });
    }

    private function moveStockTakeRecords(Product $originalProduct, Product $destinationProduct)
    {
        $this->info('Stock Take Items');
        $this->info($originalProduct->stockTakeItems()->count().' stock take items');

        $originalProduct->stockTakeItems()->eachById(function (StockTakeItem $stockTakeItem) use ($destinationProduct) {
            $this->info("$stockTakeItem->qty_counted (ID: $stockTakeItem->id)");

            $existStockTakeItem = StockTakeItem::where('product_id', $destinationProduct->id)->where('stock_take_id',
                $stockTakeItem->stock_take_id)->first();
            if (! $existStockTakeItem instanceof StockTakeItem) {
                $stockTakeItem->product_id = $destinationProduct->id;
                $stockTakeItem->save();
            } else {
                $this->handleInsufficientStockException($existStockTakeItem, $stockTakeItem->qty_counted,
                    $stockTakeItem->unit_cost);
                $stockTakeItem->delete();
            }
        });
    }

    private function moveWarehouseTransferRecords(Product $originalProduct, Product $destinationProduct)
    {
        $this->info('Warehouse Transfer Lines');
        $this->info($originalProduct->warehouseTransferLines()->count().' warehouse transfer lines');

        $originalProduct->warehouseTransferLines()->eachById(function (WarehouseTransferLine $warehouseTransferLine) use ($destinationProduct) {
            $this->info($warehouseTransferLine->warehouseTransfer->warehouse_transfer_number." @ $warehouseTransferLine->quantity (ID: $warehouseTransferLine->id)");
            $warehouseTransferLine->product_id = $destinationProduct->id;
            $warehouseTransferLine->save();
        });
    }

    private function moveBlemishedRecords(Product $originalProduct, Product $destinationProduct)
    {
        $this->info('Blemished Products');
        $this->info($originalProduct->blemishedProducts()->count().' blemished products');

        $originalProduct->blemishedProducts()->eachById(function (ProductBlemished $blemishedProduct) use ($destinationProduct) {
            $this->info($blemishedProduct->product_id." ($blemishedProduct->condition) (ID: $blemishedProduct->id)");
            $blemishedProduct->derived_from_product_id = $destinationProduct->id;
            $blemishedProduct->save();
        });
    }

    private function moveBundleRecords(Product $originalProduct, Product $destinationProduct)
    {
        $this->info('Bundles that the product belongs to');
        $this->info($originalProduct->inBundlingProducts()->count().' bundles that the product belongs to');

        $originalProduct->inBundlingProducts()->eachById(function (ProductComponent $productComponent) use ($destinationProduct) {
            $this->info("$productComponent->quantity (ID: $productComponent->id)");
            $productComponent->component_product_id = $destinationProduct->id;
            $productComponent->save();
        });
    }

    private function moveProductListingRecords(Product $originalProduct, Product $destinationProduct)
    {
        /*
         * TODO: May need a cache update
         * TODO: May want to present user with option to delete listing?
         *
         * TODO: BAB: Need to update product_listings_bk table on dev?
         */
        $this->info('Product Listings');
        $this->info($originalProduct->productListings()->count().' product listings');

        $originalProduct->productListings()->eachById(function (ProductListing $productListing) use ($destinationProduct, &$integrationInstances) {
            $this->info("$productListing->sales_channel_listing_id on ".$productListing->salesChannel->name." (ID: $productListing->id)");
            $productListing->product_id = $destinationProduct->id;
            $productListing->save();

            $this->integrationInstances[] = $productListing->salesChannel->integrationInstance;
        });
    }

    /**
     * Create a new inventory adjustment to cover the quantity of the link
     *
     * @param  SalesOrderLine|InventoryAdjustment|StockTakeItem  $link
     *
     * @throws \Exception
     */
    private function handleInsufficientStockException($link, int $quantity, ?float $unitCost = null): void
    {
        // a new inventory adjustment from sales order line data
        $inventoryAdjustment = new InventoryAdjustment();
        $inventoryAdjustment->adjustment_date = now();
        $inventoryAdjustment->product_id = $link->product_id;
        $inventoryAdjustment->warehouse_id = $link instanceof StockTakeItem ? $link->stockTake->warehouse_id : $link->warehouse_id;
        $inventoryAdjustment->quantity = $quantity;
        $inventoryAdjustment->notes = 'sku.io integrity, deprecate old sku';
        $inventoryAdjustment->link_id = $link->id;
        $inventoryAdjustment->link_type = get_class($link);
        $inventoryAdjustment->save();

        // add to stock without releasing backorder queues to cover the quantity of the link that caused the insufficient stock exception
        InventoryManager::with(
            $inventoryAdjustment->warehouse_id,
            $inventoryAdjustment->product
        )->addToStock(abs($inventoryAdjustment->quantity), $inventoryAdjustment, true, false, $unitCost);
    }
}
