<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;

class MonitorInnodbLocksCommand extends Command
{
    protected $signature = 'sku:monitor-innodb-locks';

    protected $description = 'Monitor InnoDB locks';

    public function handle(): void
    {
        // Query INNODB_LOCK_WAITS table
        $lockWaits = DB::select('
            SELECT rt.trx_id AS requesting_trx_id,
                   rt.trx_mysql_thread_id AS requesting_mysql_thread,
                   rt.trx_query AS requesting_query,
                   rl.lock_mode AS requesting_lock_mode,
                   rl.lock_type AS requesting_lock_type,
                   rl.lock_index AS requesting_lock_index,
                   rl.lock_data AS requesting_lock_data,
                   bt.trx_id AS blocking_trx_id,
                   bt.trx_mysql_thread_id AS blocking_mysql_thread,
                   bt.trx_query AS blocking_query,
                   bl.lock_mode AS waiting_lock_mode,
                   bl.lock_type AS waiting_lock_type,
                   bl.lock_index AS waiting_lock_index,
                   bl.lock_data AS waiting_lock_data,
                   TIMESTAMPDIFF(SECOND, rt.trx_started, NOW()) requesting_waiting_time_seconds,
                   TIMESTAMPDIFF(SECOND, bt.trx_started, NOW()) blocking_time_seconds
            FROM information_schema.INNODB_LOCK_WAITS AS w 
            INNER JOIN information_schema.INNODB_LOCKS AS rl ON w.requesting_trx_id = rl.lock_trx_id
            INNER JOIN information_schema.INNODB_TRX AS rt ON rt.trx_id = w.requesting_trx_id
            INNER JOIN information_schema.INNODB_LOCKS AS bl ON w.blocking_trx_id = bl.lock_trx_id
            INNER JOIN information_schema.INNODB_TRX AS bt ON bt.trx_id = w.blocking_trx_id
');

        // If there are any lock waits, dump the INNODB_TRX table
        if (! empty($lockWaits)) {
            $transactions = DB::select('SELECT * FROM information_schema.INNODB_TRX');
            customlog('innodb_locks', 'Detected InnoDB Lock Wait!
            
            =========================Lock Waits w/ Extended Info:=========================
            '.json_encode($lockWaits, JSON_PRETTY_PRINT).'
            =========================All Locks:=========================
            '.json_encode(DB::select('SELECT * FROM information_schema.INNODB_LOCKS'), JSON_PRETTY_PRINT).'
            =========================All Transactions:=========================
            '.json_encode($transactions, JSON_PRETTY_PRINT).'
            =========================Processes:=========================
            '.json_encode(DB::select('SELECT * FROM information_schema.PROCESSLIST'), JSON_PRETTY_PRINT));
        }
    }
}
