<?php

namespace App\Console\Commands;

use App\Data\CreateInitialStockTakeData;
use App\Data\CreateInitialStockTakeItemData;
use App\Helpers;
use App\Models\FifoLayer;
use App\Models\InventoryMovement;
use App\Models\Setting;
use App\Models\StockTakeItem;
use App\Models\Warehouse;
use App\Services\StockTake\StockTakeManager;
use Exception;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Modules\Amazon\Entities\AmazonFbaInitialInventory;
use Throwable;

class MigrateInitialCountsToStockTakeCommand extends Command
{
    protected $signature = 'sku:patch:migrate-initial-counts-to-stock-takes';

    protected $description = 'Migrate Initial Counts to Stock Takes';

    /**
     * @throws Throwable
     */
    public function handle(): void
    {
        $this->info('Migrating initial counts to stock takes...');

        $manager = app(StockTakeManager::class);

        DB::transaction(function () use ($manager) {
            $movementsByWarehouse = InventoryMovement::with(['layer.inventoryMovements'])
                ->where('type', 'initial_count')
                ->get()
                ->groupBy('warehouse_id');

            foreach ($movementsByWarehouse as $warehouseId => $warehouseMovements) {
                $warehouse = Warehouse::find($warehouseId);

                if ($warehouse->type == Warehouse::TYPE_SUPPLIER) {
                    throw new Exception('Warehouse type supplier not supported for initial count migration.');
                }

                $this->info('Migrating for warehouse ' . $warehouse->name . '...');

                if ($warehouse->initialCountStockTake) {
                    dd("Warehouse {$warehouse->name} already has an initial count stock take.");
                }

                $stockTakeItemsData = collect();

                $warehouseMovements->each(function (InventoryMovement $movement) use ($stockTakeItemsData) {
                    /** @var FifoLayer $fifoLayer */
                    $fifoLayer = $movement->layer;
                    $this->info('Migrating initial counts to stock takes for fifo layer ' . $fifoLayer->id);
                    if ($item = $stockTakeItemsData->where('product_id', $movement->product_id)->first()) {
                        $this->info("Found existing stock take item for product {$movement->product_id}. Adding quantity $movement->quantity.");
                        $item->qty_counted += $movement->quantity;
                        $stockTakeItemsData->push($item);
                    }
                    else {
                        $this->info("Creating new stock take item for product {$movement->product_id}. Qty: $movement->quantity.");
                        $stockTakeItemsData->push(CreateInitialStockTakeItemData::from([
                            'product_id' => $movement->product_id,
                            'qty_counted' => $movement->quantity,
                            'unit_cost' => $fifoLayer->avg_cost,
                        ]));
                    }
                });

                $this->info('Creating initial stock take for warehouse ' . $warehouse->name . '...');
                $stockTake = $manager->createInitialStockTake(CreateInitialStockTakeData::from([
                    'date_count' => Helpers::setting(Setting::KEY_INVENTORY_START_DATE),
                    'warehouse_id' => $warehouse->id,
                    'items' => $stockTakeItemsData,
                ]), false);

                $warehouseMovements->each(function (InventoryMovement $originalInitialCountMovement) use ($stockTake)
                {
                    /** @var StockTakeItem $matchingStockTakeItem */
                    $matchingStockTakeItem = $stockTake->stockTakeItems->where('product_id', $originalInitialCountMovement->product_id);
                    if ($matchingStockTakeItem->count() != 1) {
                        throw new Exception("Stock take item not found for product {$originalInitialCountMovement->product_id}");
                    }

                    $matchingStockTakeItem = $matchingStockTakeItem->first()->refresh();

                    $this->info("Deleting original initial count movement {$originalInitialCountMovement->id}, since it was created as a stock take item movement {$matchingStockTakeItem->getOriginatingMovement()->id}.");
                    $originalInitialCountMovement->delete();

                    $stockTakeFifoLayer = $matchingStockTakeItem->creationFifoLayer;

                    // Move original fifo layer usages to new layer
                    /** @var FifoLayer $originalFifoLayer */
                    $originalFifoLayer = $originalInitialCountMovement->layer;

                    $originalFifoLayer->refresh()->inventoryMovements()->orderBy('quantity', 'desc')->each(function (InventoryMovement $originalMovement) use ($stockTakeFifoLayer, $originalFifoLayer, $matchingStockTakeItem) {
                        $originalMovement->layer_id = $stockTakeFifoLayer->id;
                        $originalMovement->save();
                        if ($originalMovement->inventory_status == 'active' && $originalMovement->quantity < 0) {
                            $this->info("Migrating usage {$originalMovement->id} (qty: $originalMovement->quantity) to stock take fifo layer {$stockTakeFifoLayer->id} from fifo layer {$originalFifoLayer->id} (qty original: {$stockTakeFifoLayer->original_quantity}, qty fulfilled: {$stockTakeFifoLayer->fulfilled_quantity}, qty available: {$stockTakeFifoLayer->available_quantity}).");
                            $stockTakeFifoLayer->fulfilled_quantity += abs($originalMovement->quantity);
                            $stockTakeFifoLayer->save();
                        }
                    });

                    if ($amazonInitialInventory = AmazonFbaInitialInventory::where('fifo_layer_id', $originalFifoLayer->id)->first()) {
                        $amazonInitialInventory->fifo_layer_id = $stockTakeFifoLayer->id;
                        $amazonInitialInventory->save();
                    }
                    $originalFifoLayer->delete();
                });
            }
        });

        $this->info('Initial counts migrated to stock takes successfully.');
    }
}
