<?php

namespace App\Console\Commands\Magento;

use App\Helpers\YamlHelper;
use App\Models\IntegrationInstance;
use Carbon\Carbon;
use Illuminate\Console\Command;

class GetStores extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'sku:magento:get-stores 
                          {integrationInstance : The ID of the integration instance}
                          {--createdAfter= : Return products created after a specified date. }
                          {--createdBefore= : Return products created before a specified date. }
                          {--lastUpdatedAfter= : Return products last updated after a specified date. }
                          {--lastUpdatedBefore= : Return products last updated before a specified date. }
                          {--l|limit=250 : Return up to this many results per page. }
                          {--debug : Run the job synchronously instead of queueing it for the purpose of debugging}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Get magento stores for the given integration instance.';

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        /** @var IntegrationInstance $integrationInstance */
        $integrationInstance = IntegrationInstance::query()
            ->findOrFail($this->argument('integrationInstance'));

        if (! $integrationInstance || ! $integrationInstance->isMagento()) {
            $this->error('The integration must be Magento.');

            return 0;
        }

        $job = new \App\Jobs\Magento\GetStores($integrationInstance, $this->getMagentoOptions());

        if ($this->option('debug')) {
            $result = $job->handle();
            $this->info(YamlHelper::dump($result));
        } else {
            dispatch($job)->onQueue('sales-channels');
            $this->info('Added Magento stores request to the Queue, it will be processed shortly.');
        }
    }

    /**
     * Parse command options to Magento searchCriteria.
     *
     * @see https://devdocs.magento.com/guides/v2.4/rest/performing-searches.html
     */
    protected function getMagentoOptions(): array
    {
        $options = [];
        if ($this->option('createdAfter')) {
            $options['filterGroups'][]['filters'][0] = [
                'field' => 'created_at',
                'conditionType' => 'gt',
                'value' => Carbon::parse($this->option('createdAfter'))->toDateTimeString(),
            ];
        }
        if ($this->option('createdBefore')) {
            $options['filterGroups'][]['filters'][0] = [
                'field' => 'created_at',
                'conditionType' => 'lt',
                'value' => Carbon::parse($this->option('createdBefore'))->toDateTimeString(),
            ];
        }
        if ($this->option('lastUpdatedAfter')) {
            $options['filterGroups'][]['filters'][0] = [
                'field' => 'updated_at',
                'conditionType' => 'gt',
                'value' => Carbon::parse($this->option('lastUpdatedAfter'))->toDateTimeString(),
            ];
        }
        if ($this->option('lastUpdatedBefore')) {
            $options['filterGroups'][]['filters'][0] = [
                'field' => 'updated_at',
                'conditionType' => 'lt',
                'value' => Carbon::parse($this->option('lastUpdatedBefore'))->toDateTimeString(),
            ];
        }

        if ($this->option('limit')) {
            $options['pageSize'] = (int) $this->option('limit');
        } else {
            $options['pageSize'] = (int) 300;
        }

        return $options;
    }
}
