<?php

namespace App\Console\Commands\Magento;

use App\Enums\DownloadedBy;
use App\Jobs\Magento\GetOrdersJob;
use App\Models\IntegrationInstance;
use Illuminate\Console\Command;

class GetOrdersCommand extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'sku:magento:get-orders
                                {integrationInstance : The ID of the integration instance}
                                {--createdAfter= : A date used for selecting orders created after a specified time.  }
                                {--createdBefore= : A date used for selecting orders created before a specified time.  }
                                {--lastUpdatedAfter= : A date used for selecting orders that were last updated after a specified time.  }
                                {--lastUpdatedBefore= : A date used for selecting orders that were last updated before a specified time.  }
                                {--s|status=* : A list of order status values. }
                                {--l|limit=250 : A number that indicates the maximum number of orders that can be returned per page. }
                                {--i|ids=* : Specify order increment ids }
                                {--debug : Do not queue, run inline }';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Retrieve Orders from Magento sales channel';

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        /** @var IntegrationInstance $integrationInstance */
        $integrationInstance = $this->argument('integrationInstance') instanceof IntegrationInstance ? $this->argument('integrationInstance') : IntegrationInstance::with(['integration'])->findOrFail($this->argument('integrationInstance'));

        if (! $integrationInstance->isMagento()) {
            $this->error('The integration instance type must be Magento');

            return 0;
        }

        $options = array_merge(['downloaded_by' => DownloadedBy::Command], $this->options());

        if ($this->option('ids') || $this->option('debug')) {
            dispatch_sync(new GetOrdersJob($integrationInstance, $options));
        } else {
            $this->info('Dispatching the "GetOrders" Magento job');
            dispatch(new GetOrdersJob($integrationInstance, $options))->onQueue('sales-channels');
        }
    }
}
