<?php

namespace App\Console\Commands\Inventory\Patches;

use App\Exceptions\InsufficientStockException;
use App\Exceptions\InventoryMovementTypeException;
use App\Models\InventoryAdjustment;
use App\Models\InventoryMovement;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderFulfillmentLine;
use App\Models\SalesOrderLine;
use App\Services\InventoryManagement\InventoryManager;
use App\Services\SalesOrder\Fulfillments\FulfillmentManager;
use App\Services\StockTake\OpenStockTakeException;
use Illuminate\Console\Command;
use Illuminate\Database\Query\Builder;
use Illuminate\Support\Facades\DB;
use Throwable;

class PortOverFulfillmentsToNegativeAdjustments extends Command
{

    protected $signature = 'sku:inventory:patch:port-over-fulfillments-to-negative-adjustments
                                {--i|id=* : Specify product ids }';

    protected $description = 'Port over fulfillments to negative adjustments';


    public function handle(): int
    {
        $query = $this->getOverFulfillmentsQuery();

        $query->cursor()
            ->each(/**
             * @throws Throwable
             */ function ($result) {
                $this->info("Fixing {$result->ref1} with Product ID: $result->product_id");
                DB::beginTransaction();

                try {
                    $this->fixForResult($result);
                    DB::commit();

                    $this->info("{$result->ref1}: $result->product_id Fixed");
                } catch (Throwable $e) {
                    DB::rollBack();
                    $this->error("{$result->ref1}: $result->product_id Failed");
                    $this->error($e->getMessage());
                }
            });

        return 0;
    }

    /**
     * @param $result
     * @return void
     * @throws InventoryMovementTypeException
     * @throws OpenStockTakeException
     * @throws Throwable
     * @throws InsufficientStockException
     */
    private function fixForResult($result): void
    {
        $reductionQty = (int)($result->sol_total - $result->fulfillment_total);

        if($reductionQty <= 0){
            return;
        }

        // Update the fulfillment line
        /** @var SalesOrderFulfillmentLine $fulfillmentLine */
        $fulfillmentLine = SalesOrderFulfillmentLine::with(['salesOrderFulfillment', 'salesOrderLine'])
            ->findOrFail($result->sofl_id);

        $fulfillmentLine->quantity -= $reductionQty;
        $fulfillmentLine->save();

        // Update fulfilled quantity on sales order line.
        $fulfillmentLine->salesOrderLine->fulfilled_quantity -= $reductionQty;
        $fulfillmentLine->salesOrderLine->save();

        // Update inventory movement quantity.
        /** @var InventoryMovement $fulfillmentMovement */
        $fulfillmentMovement = InventoryMovement::query()
            ->findOrFail($result->fulfillment_movement_id);
        $fulfillmentMovement->quantity += $reductionQty; // Note that fulfillment movement qty is negative.
        $fulfillmentMovement->save();

        $fulfillmentLine->salesOrderFulfillment->update([
            'status' => SalesOrderFulfillment::STATUS_FULFILLED
        ]);

        if($fulfillmentLine->quantity <= 0){
            $fulfillmentLine->deleteWithFulfillmentIfLast(true);
        }

        $note = "$reductionQty unit(s) of over-fulfillment on fulfillment: {$fulfillmentLine->salesOrderFulfillment->fulfillment_number} has been treated as negative adjustment.";

        // Create negative adjustment
        /** @var InventoryAdjustment $adjustment */
        $adjustment = InventoryAdjustment::query()->create([
            'adjustment_date' => now(),
            'product_id' => $result->product_id,
            'quantity' => -$reductionQty,
            'notes' => $note,
            'reference' => $result->ref1,
            'link_type' => SalesOrderLine::class,
            'link_id' => $fulfillmentLine->sales_order_line_id,
            'reason' => InventoryAdjustment::ADJUSTMENT_REASON_OVER_FULFILLMENT,
            'warehouse_id' => $fulfillmentLine->salesOrderLine->warehouse_id,
        ]);

        $salesOrderLine = $fulfillmentLine->salesOrderLine;
        $manager = InventoryManager::with(
            warehouseId: $salesOrderLine->warehouse_id,
            product: $salesOrderLine->product
        );
        $manager->takeFromStock(
            quantity: $reductionQty,
            event: $adjustment,
            excludedSources: [$salesOrderLine->id]
        );


        // Refresh fulfillment status on sales order.
        $salesOrderLine->salesOrder->updateFulfillmentStatus(
            fulfilledDate: $fulfillmentLine->salesOrderFulfillment->fulfilled_at
        );

        // Add note to the sales order
        $salesOrderLine->salesOrder->notes()
            ->create([
                'note' => $note
            ]);

    }

    private function getOverFulfillmentsQuery(): Builder
    {
        $fulfillmentLineMovementsQuery = InventoryMovement::with([])
            ->selectRaw('
                im.id fulfillment_movement_id, sum(im.quantity) fulfillment_total, 
                REGEXP_replace(im.reference, "\\\.[1-9]","") ref1,
                im.product_id, sofl.id AS sofl_id, sofl.sales_order_line_id
            ')
            ->from('inventory_movements as im')
            ->join(
                'sales_order_fulfillment_lines as sofl',
                'sofl.id',
                'im.link_id'
            )
            ->where('im.link_type', SalesOrderFulfillmentLine::class)
            ->groupByRaw('REGEXP_replace(im.reference, "\\\.[1-9]",""), sofl.sales_order_line_id');

        $salesOrderLineMovementsQuery = InventoryMovement::with([])
            ->selectRaw('
                im2.id, sum(im2.quantity) sol_total, 
                REGEXP_replace(im2.reference, "\\\.[1-9]","") ref2, 
                sol.id AS sol_id
            ')
            ->from('inventory_movements as im2')
            ->join(
                'sales_order_lines as sol',
                'sol.id',
                'im2.link_id'
            )
            ->where('im2.link_type', SalesOrderLine::class)
            ->where('im2.inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
            ->groupByRaw('REGEXP_replace(im2.reference, "\\\.[1-9]",""), sol.id');


        $query = DB::query()
            ->select('*')
            ->fromSub($fulfillmentLineMovementsQuery, 'flm')
            ->leftJoinSub($salesOrderLineMovementsQuery, 'slm', function ($join) {
                $join->on('slm.sol_id', '=', 'flm.sales_order_line_id');
            })
            ->whereColumn('slm.sol_total', '>', 'flm.fulfillment_total')
            ->orWhereNull('slm.sol_total');

        if (! empty($this->option('id'))) {
            $query->whereIn('flm.product_id', $this->option('id'));
        }

        return $query;
    }
}