<?php

namespace App\Console\Commands\Inventory\Patches;

use App\Models\FifoLayer;
use App\Models\InventoryMovement;
use App\Models\SalesOrder;
use Illuminate\Console\Command;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\DB;
use Throwable;

class MapFulfillmentMovementsToRightLayers extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'sku:inventory:patch:map-fulfillment-movements-to-right-layers
                                {--i|ids=* : Specify product ids }';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Maps inventory movements of fulfillments to the same layers on the original reservations.';

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        $query = InventoryMovement::with([])
            ->selectRaw('layer_id, reference, sum(quantity) as total_qty, sku, product_id')
            ->join('products as p', 'p.id', 'inventory_movements.product_id')
            ->join('sales_orders as so', 'so.sales_order_number', 'inventory_movements.reference')
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_RESERVED)
            ->where('layer_type', FifoLayer::class)
            ->where('so.order_status', SalesOrder::STATUS_CLOSED);

        if (! empty($this->option('ids'))) {
            $query->whereIn('product_id', $this->option('ids'));
        }

        $query->groupBy(['layer_id', 'reference', 'product_id', 'sku'])
            ->having('total_qty', '<', 0)
            ->orderBy('reference');

        $query->cursor()
            ->each(function ($result) {
                $this->info("Fixing {$result->reference}: $result->sku");
                DB::beginTransaction();

                try {
                    $this->fixForResult($result);
                    DB::commit();

                    $this->info("{$result->reference}: $result->sku Fixed");
                } catch (Throwable $e) {
                    DB::rollBack();
                    $this->error("{$result->reference}: $result->sku not fixed: {$e->getMessage()}");
                }
            });

        return 0;
    }

    private function fixForResult($result)
    {
        $layerId = $result->layer_id;
        $reference = $result->reference;
        $productId = $result->product_id;
        $sku = $result->sku;

        $thisReservationQuery = InventoryMovement::with([])
            ->where('reference', $reference)
            ->where('product_id', $productId)
            ->where('layer_type', FifoLayer::class)
            ->where('layer_id', $layerId)
            ->where('quantity', '<', 0)
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_RESERVED);

        if ($thisReservationQuery->count() > 1) {
            $this->handleMultipleExistingReservations($thisReservationQuery);

            return;
        }

        /** @var InventoryMovement $thisReservation */
        $thisReservation = $thisReservationQuery->first();

        // Get the original reservation
        $originalReservationQuery = InventoryMovement::with([])
            ->where('reference', $reference)
            ->where('product_id', $productId)
            ->where('layer_type', FifoLayer::class)
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_RESERVED)
            ->where('quantity', '>', 0);
        if ($originalReservationQuery->count() > 1) {
            // We get the one without negative reservation and has quantity of
            // at least the quantity on the negative reservation that needs fixing.
            $originalReservation = $this->getMatchingReservation($originalReservationQuery, $thisReservation, $layerId);
        } else {
            /** @var InventoryMovement $originalReservation */
            $originalReservation = $originalReservationQuery->first();
        }

        if (! $originalReservation) {
            $this->warn('No existing reservation to match to.');

            return;
        }

        if ($originalReservation->quantity == abs($thisReservation->quantity)) {
            $thisReservation->update(['layer_id' => $originalReservation->layer_id]);
        }
    }

    private function getMatchingReservation(Builder $originalReservationQuery, InventoryMovement $thisReservation): ?InventoryMovement
    {
        foreach ($originalReservationQuery->get() as $movement) {
            if ($this->movementHasNegativeReservation($movement)) {
                continue;
            }
            if ($movement->quantity >= abs($thisReservation->quantity)) {
                // Movement has no negative reservation and has enough
                // quantity to map to the negative reservation for the fulfillment.
                return $movement;
            }
        }

        return null;
    }

    private function movementHasNegativeReservation(InventoryMovement $movement): bool
    {
        return InventoryMovement::with([])
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_RESERVED)
            ->where('reference', $movement->reference)
            ->where('product_id', $movement->product_id)
            ->where('quantity', '<', 0)
            ->where('layer_type', FifoLayer::class)
            ->where('layer_id', $movement->layer_id)
            ->count() > 0;
    }

    private function handleMultipleExistingReservations(Builder $thisReservationQuery)
    {
        $usedMovementsIds = [];
        /** @var InventoryMovement $reservation */
        foreach ($thisReservationQuery->get() as $reservation) {
            /** @var InventoryMovement $matchedReservation */
            $matchedReservation = InventoryMovement::with([])
                ->where('product_id', $reservation->product_id)
                ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_RESERVED)
                ->where('quantity', abs($reservation->quantity))
                ->where('layer_type', FifoLayer::class)
                ->where('warehouse_id', $reservation->warehouse_id)
                ->where('reference', $reservation->reference)
                ->whereNotIn('id', $usedMovementsIds)
                ->first();
            if ($matchedReservation) {
                if (! $this->movementHasNegativeReservation($matchedReservation) && $matchedReservation->quantity >= abs($reservation->quantity)) {
                    $reservation->update(['layer_id' => $matchedReservation->layer_id]);
                }
                $usedMovementsIds[] = $matchedReservation->id;
            }
        }
    }
}
